import sys
import os
import configparser
from PyQt5.QtWidgets import (
    QApplication,
    QWidget,
    QPushButton,
    QFileDialog,
    QDialog,
    QLabel,
    QLineEdit,
    QHBoxLayout,
    QVBoxLayout,
    QColorDialog,
    QDesktopWidget,
    QMessageBox
)
# QDesktopWidgetは非推奨ですが、Qt5の環境互換のためここではそのまま使用
from PyQt5.QtCore import Qt, QRect, QPoint, pyqtSignal
from PyQt5.QtGui import QPainter, QPen, QColor

# --- 0. 設定ファイル管理 ---

CONFIG_FILE = "setting.ini"

# デフォルト設定値 (変更なし)
DEFAULT_SETTINGS = {
    'Geometry': {
        'width': '800',
        'height': '800',
    },
    'Appearance': {
        'color': '#FF0000', # 赤
    },
    'LastSave': {
        'directory': os.path.expanduser("~"),
    }
}

# load_settings関数 (変更なし)
def load_settings():
    """設定ファイルから設定値を読み込む。ファイルが存在しない場合はデフォルト値を返す。"""
    config = configparser.ConfigParser()
    
    # デフォルト値を設定
    for section, values in DEFAULT_SETTINGS.items():
        if section not in config:
            config.add_section(section)
        for key, value in values.items():
            config.set(section, key, str(value))
    
    # ファイルの読み込み
    config.read(CONFIG_FILE, encoding='utf-8')
    
    try:
        settings = {
            'width': int(config.get('Geometry', 'width')),
            'height': int(config.get('Geometry', 'height')),
            'color': config.get('Appearance', 'color'),
            'last_dir': config.get('LastSave', 'directory')
        }
        # ディレクトリが存在しない場合のフォールバック
        if not os.path.isdir(settings['last_dir']):
             settings['last_dir'] = os.path.expanduser("~")
            
        return settings
    except (configparser.NoSectionError, configparser.NoOptionError, ValueError):
        print(f"設定ファイル '{CONFIG_FILE}' の読み込みエラーまたはファイルが存在しないため、デフォルト値を使用します。")
        return {
            'width': int(DEFAULT_SETTINGS['Geometry']['width']),
            'height': int(DEFAULT_SETTINGS['Geometry']['height']),
            'color': DEFAULT_SETTINGS['Appearance']['color'],
            'last_dir': DEFAULT_SETTINGS['LastSave']['directory']
        }

# save_settings関数 (変更なし)
def save_settings(width, height, color_hex, last_dir):
    """現在の設定値を設定ファイルに書き込む。"""
    config = configparser.ConfigParser()
    config.read(CONFIG_FILE, encoding='utf-8')
    
    # 既存の設定をロードし、上書きする（存在しない場合はデフォルトから作成）
    if 'Geometry' not in config: config.add_section('Geometry')
    config.set('Geometry', 'width', str(width))
    config.set('Geometry', 'height', str(height))
    
    if 'Appearance' not in config: config.add_section('Appearance')
    config.set('Appearance', 'color', color_hex)
    
    if 'LastSave' not in config: config.add_section('LastSave')
    config.set('LastSave', 'directory', last_dir)
    
    try:
        with open(CONFIG_FILE, 'w', encoding='utf-8') as configfile:
            config.write(configfile)
    except Exception as e:
        print(f"❌ 設定ファイルの保存に失敗しました: {e}")


# --- 1. 設定ウィンドウ ---
class SettingsDialog(QDialog):
    def __init__(self, default_width=800, default_height=800, default_color="#FF0000"):
        super().__init__()
        self.setWindowTitle("キャプチャ設定")
        
        self.width_val = int(default_width)
        self.height_val = int(default_height)
        self.color_val = QColor(default_color)

        self.init_ui()
        
    def init_ui(self):
        main_layout = QVBoxLayout(self)

        # --- サイズ設定 ---
        size_group = QHBoxLayout()
        # 幅 (Width)
        size_group.addWidget(QLabel("幅 (px):"))
        self.width_input = QLineEdit(str(self.width_val))
        self.width_input.setFixedWidth(80)
        size_group.addWidget(self.width_input)
        
        # 高さ (Height)
        size_group.addWidget(QLabel("高さ (px):"))
        self.height_input = QLineEdit(str(self.height_val))
        self.height_input.setFixedWidth(80)
        size_group.addWidget(self.height_input)
        
        main_layout.addLayout(size_group)

        # --- 色設定 ---
        color_group = QHBoxLayout()
        color_group.addWidget(QLabel("ラインの色:"))
        
        self.color_display = QLabel("■")
        self.color_display.setFixedSize(20, 20)
        self.color_display.setStyleSheet(f"background-color: {self.color_val.name()}; font-size: 16px; color: {self.color_val.name()};")
        color_group.addWidget(self.color_display)
        
        self.color_button = QPushButton("色の選択")
        self.color_button.clicked.connect(self.choose_color)
        color_group.addWidget(self.color_button)
        color_group.addStretch(1)
        
        main_layout.addLayout(color_group)

        # --- 確定ボタン ---
        # ⭐ ボタンテキストを修正
        self.ok_button = QPushButton("設定を適用") 
        self.ok_button.clicked.connect(self.accept)
        main_layout.addWidget(self.ok_button)

    def choose_color(self):
        color = QColorDialog.getColor(self.color_val, self, "ラインの色の選択")
        if color.isValid():
            self.color_val = color
            self.color_display.setStyleSheet(f"background-color: {self.color_val.name()}; font-size: 16px; color: {self.color_val.name()};")

    def accept(self):
        try:
            width = int(self.width_input.text())
            height = int(self.height_input.text())
            
            if width <= 0 or height <= 0:
                 raise ValueError("サイズは正の整数である必要があります。")
            
            self.width_val = width
            self.height_val = height
            
            # 設定値を設定ファイルに保存 (最終保存ディレクトリは変更しない)
            current_settings = load_settings()
            save_settings(self.width_val, self.height_val, self.color_val.name(), current_settings['last_dir'])
            
            super().accept()
            
        except ValueError as e:
            print(f"入力エラー: {e}")
            self.width_input.setText(str(self.width_val))
            self.height_input.setText(str(self.height_val))

    def get_settings(self):
        return self.width_val, self.height_val, self.color_val.name() # 色をHEX文字列で返す

# ---------------------------------------------------------------------------------------------------

# --- 2. メインのオーバーレイウィンドウ ---
class OverlayWindow(QWidget):
    # ⭐ 1. シグナルを定義: 新しい設定値 (幅, 高さ, 色) を渡す
    restart_signal = pyqtSignal(int, int, str)

    def __init__(self, rect_width, rect_height, line_color, last_save_dir):
        super().__init__()
        
        # 設定値の保持 (QColorオブジェクトで保持)
        self.RECT_WIDTH = rect_width
        self.RECT_HEIGHT = rect_height
        # str型のcolor (HEX) をQColorオブジェクトに変換
        if isinstance(line_color, str):
            self.LINE_COLOR = QColor(line_color)
        else:
            self.LINE_COLOR = line_color
            
        self.LAST_SAVE_DIR = last_save_dir 
        
        self._drag_pos = QPoint()
        
        # 画面サイズとウィンドウ位置の初期計算 (中央に配置)
        self.center_window(self.RECT_WIDTH, self.RECT_HEIGHT)

        # ウィンドウのスタイルを設定
        self.setWindowFlags(
            Qt.FramelessWindowHint |
            Qt.WindowStaysOnTopHint |
            Qt.Tool
        )
        
        self.setAttribute(Qt.WA_TranslucentBackground)
        
        # ボタンの配置とレイアウト
        self.init_ui()
        
        print("-" * 30)
        print(f"起動しました: 幅={self.RECT_WIDTH}, 高さ={self.RECT_HEIGHT}, 色={self.LINE_COLOR.name()}")
        print(f"最終保存先: {self.LAST_SAVE_DIR}")

    def center_window(self, width, height):
        screen = QApplication.primaryScreen().geometry()
        screen_w = screen.width()
        screen_h = screen.height()
        
        self.window_x = (screen_w - width) // 2
        self.window_y = (screen_h - height) // 2
        
        self.setGeometry(self.window_x, self.window_y, width, height)

    def init_ui(self):
        """UIコンポーネントの初期化と配置"""
        
        # Capture ボタン (省略)
        self.capture_button = QPushButton("Capture", self)
        self.capture_button.setFixedSize(50, 25)
        self.capture_button.setStyleSheet("""QPushButton {background-color: #0078D4; color: white; border: 2px solid #005A9E; border-radius: 5px;} QPushButton:hover {background-color: #005A9E;}""")
        self.capture_button.clicked.connect(self.capture_and_save_dialog)
        
        # Setting ボタン
        self.setting_button = QPushButton("Setting", self)
        self.setting_button.setFixedSize(50, 25)
        self.setting_button.setStyleSheet("""QPushButton {background-color: #009900; color: white; border: 2px solid #005A9E; border-radius: 5px;} QPushButton:hover {background-color: #005A9E;}""")
        # ⭐ 接続: 設定変更シグナルを発行するメソッドに接続
        self.setting_button.clicked.connect(self.change_setting) 
        
        # 終了ボタン (省略)
        self.exit_button = QPushButton("Close", self)
        self.exit_button.setFixedSize(50, 25)
        self.exit_button.setStyleSheet("""QPushButton {background-color: #CC0000; color: white; border: 2px solid #990000; border-radius: 5px;} QPushButton:hover {background-color: #990000;}""")
        self.exit_button.clicked.connect(self.close_application_immediately)
        
        # レイアウト (省略)
        button_layout = QHBoxLayout()
        button_layout.setSpacing(0) 
        button_layout.addStretch(1)
        button_layout.addWidget(self.capture_button)
        button_layout.addSpacing(10)
        button_layout.addWidget(self.setting_button)
        button_layout.addSpacing(10)
        button_layout.addWidget(self.exit_button)
        button_layout.addStretch(1)
        
        layout = QVBoxLayout(self)
        layout.addStretch(1) 
        layout.addLayout(button_layout) 
        layout.addStretch(1)
        
        self.setLayout(layout)
        
        self.activateWindow()
        self.raise_()

    # ⭐ 2. 設定変更時の処理を修正 (シグナル発行)
    def change_setting(self):
        """Settingボタンが押されたときに設定ダイアログを表示し、結果をシグナルで通知する"""
        
        # 現在の設定値を取得してダイアログに渡す
        settings_dialog = SettingsDialog(
            default_width=self.RECT_WIDTH,
            default_height=self.RECT_HEIGHT,
            default_color=self.LINE_COLOR.name()
        )
        
        # 修正: settings_dialogの親をself (OverlayWindow) に設定することで、
        #       OverlayWindowの前面にダイアログが表示されるようにする
        settings_dialog.setParent(self, settings_dialog.windowFlags()) # 親を設定
        
        if settings_dialog.exec_() == QDialog.Accepted:
            # ユーザーが「設定を適用」を押した場合、新しい設定値を取得
            new_width, new_height, new_color_str = settings_dialog.get_settings()
            
            # シグナルを発行して、メインループにウィンドウ再構築を指示する
            self.restart_signal.emit(new_width, new_height, new_color_str)
            
        else:
            print("設定変更がキャンセルされました。")

    def close_application_immediately(self):
        """確認なしで、アプリケーションを即座に終了する"""
        print("アプリケーションを終了します。")
        self.close()
        QApplication.quit()

    def paintEvent(self, event):
        """設定されたサイズと色の矩形のラインを描画"""
        painter = QPainter(self)
        
        pen = QPen(self.LINE_COLOR) # self.LINE_COLORを使用
        pen.setWidth(4)
        painter.setPen(pen)
        painter.setBrush(Qt.NoBrush)
        
        rect = QRect(0, 0, self.width() - 1, self.height() - 1)
        painter.drawRect(rect)
        
        painter.end()
        
    # --- ドラッグ移動機能、キャプチャ機能 (変更なし) ---
    def mousePressEvent(self, event):
        if event.button() == Qt.LeftButton:
            self._drag_pos = event.globalPos() - self.frameGeometry().topLeft()
            event.accept()

    def mouseMoveEvent(self, event):
        if event.buttons() == Qt.LeftButton:
            self.move(event.globalPos() - self._drag_pos)
            event.accept()
            self.window_x = self.frameGeometry().x()
            self.window_y = self.frameGeometry().y()

    def capture_screen_region(self):
        capture_rect = QRect(self.window_x, self.window_y, self.RECT_WIDTH, self.RECT_HEIGHT)
        screen = QApplication.primaryScreen()
        desktop = QApplication.desktop()
        if desktop:
            screenshot = screen.grabWindow(
                desktop.winId(), 
                capture_rect.x(), 
                capture_rect.y(), 
                capture_rect.width(), 
                capture_rect.height()
            )
        else:
            screenshot = screen.grabWindow(0, capture_rect.x(), capture_rect.y(), capture_rect.width(), capture_rect.height())
        return screenshot

    def capture_and_save_dialog(self):
        self.hide()
        QApplication.processEvents()
        
        screenshot = self.capture_screen_region()
        
        self.show()

        if screenshot.isNull():
            print("❌ 画面キャプチャに失敗しました。")
            return

        initial_filename = f"capture_{self.RECT_WIDTH}x{self.RECT_HEIGHT}_{self.LINE_COLOR.name().replace('#', '')}.bmp"
        initial_path = os.path.join(self.LAST_SAVE_DIR, initial_filename)
        
        file_path, _ = QFileDialog.getSaveFileName(
            self, 
            "キャプチャを保存",
            initial_path, 
            "BMP Files (*.bmp);;PNG Files (*.png);;All Files (*)"
        )

        if file_path:
            success = screenshot.save(file_path)
            if success:
                print(f"✅ 画面キャプチャ成功！ '{file_path}' に保存されました。")
                new_dir = os.path.dirname(file_path)
                if new_dir != self.LAST_SAVE_DIR:
                    self.LAST_SAVE_DIR = new_dir
                    save_settings(self.RECT_WIDTH, self.RECT_HEIGHT, self.LINE_COLOR.name(), self.LAST_SAVE_DIR)
            else:
                print("❌ ファイル保存に失敗しました。")
        else:
            print("保存がキャンセルされました。")

# ---------------------------------------------------------------------------------------------------

# --- 3. アプリケーションのエントリーポイント (メインループ) ---

def run_application():
    """アプリケーション全体を管理するメイン関数"""
    app = QApplication(sys.argv)
    
    # ウィンドウインスタンスを保持するための変数
    global window
    window = None

    def create_and_show_window(width=None, height=None, color_str=None):
        """新しい設定でOverlayWindowを作成し表示する"""
        global window
        
        # 既存のウィンドウがあれば閉じる
        if window is not None:
            window.close()
            window.deleteLater()
            window = None
            
        # 設定をロード (直前に保存された最新の値を取得)
        current_settings = load_settings()
        
        # 引数で新しい設定値が渡された場合はそれを使用
        # (通常、width, height, color_strは設定ダイアログから渡される)
        new_width = width if width is not None else current_settings['width']
        new_height = height if height is not None else current_settings['height']
        new_color = color_str if color_str is not None else current_settings['color']
        last_save_dir = current_settings['last_dir']
        
        # 新しいOverlayWindowを作成
        window = OverlayWindow(new_width, new_height, new_color, last_save_dir)
        
        # ⭐ シグナルをハンドラに接続
        window.restart_signal.connect(create_and_show_window)
        
        window.show()

    # --- 起動時の初期設定 ---
    initial_settings = load_settings()
    settings_dialog = SettingsDialog(
        default_width=initial_settings['width'],
        default_height=initial_settings['height'],
        default_color=initial_settings['color']
    )
    
    # ダイアログが表示され、ユーザーが「設定を適用」を押すまで待機
    if settings_dialog.exec_() == QDialog.Accepted:
        # 起動時に設定値を取得 (保存済み)
        width, height, color_str = settings_dialog.get_settings()
        
        # 最初のウィンドウを作成し、メインループを開始
        create_and_show_window(width, height, color_str)
        
        sys.exit(app.exec_())
    else:
        # ユーザーがキャンセルした場合
        sys.exit(0)


if __name__ == '__main__':
    run_application()