   # -*- coding: utf-8 -*-

import sys
import os
import subprocess
import shutil
import urllib.parse
from PyQt5 import QtCore, QtGui, QtWidgets
from PyQt5.QtCore import Qt, QFileInfo, QUrl, QFile, QDir, QSettings, QTranslator, qVersion, QCoreApplication, QThread, pyqtSignal, QProcess, QProcessEnvironment, QMutex, QPointF, QRectF
from PyQt5.QtWidgets import QMainWindow, QPushButton, QApplication, QMessageBox, QLineEdit, QWidget, QFileDialog, QLabel, QProgressBar, QVBoxLayout, QAbstractItemView, QCheckBox, QGraphicsView, QGraphicsScene, QListWidget
from collections import OrderedDict
from PyQt5.QtGui import QColor, QBrush, QPolygonF, QPen, QTransform
from osgeo import ogr, osr
from pyproj import Transformer
from xml.etree import ElementTree as ET

import zipfile
import codecs
import geojson
import json
import csv
import random
import simplekml
import atexit


# ダウンロード
class DownloadWorker(QThread):
    # ダウンロード処理を別スレッドで実行するためのワーカークラス。
    finished = pyqtSignal(str, str, str, bool, str) # URL, 保存パス, ファイル名, 成功/失敗, エラーメッセージ

    def __init__(self, file_url, output_path_temp, file_name):
        super().__init__()
        self.file_url = file_url
        self.output_path_temp = output_path_temp
        self.file_name = file_name
        self._isRunning = True

    def run(self):
        # スレッドで実行されるダウンロード処理。
        try:
            process = QProcess()
            ps_download_command = f"""
                Invoke-WebRequest -Uri '{self.file_url}' -OutFile '{self.output_path_temp}'
            """
            process.start("powershell.exe", ["-NoProfile", "-ExecutionPolicy", "Unrestricted", "-Command", ps_download_command])
            process.waitForFinished(-1) # 完了を待つ
            if not self._isRunning:
                return # スレッド終了指示があった場合は中断
            exitCode = process.exitCode()
            exitStatus = process.exitStatus()
            if exitStatus == QtCore.QProcess.NormalExit and exitCode == 0:
                self.finished.emit(self.file_url, self.output_path_temp, self.file_name, True, "")
            else:
                self.finished.emit(self.file_url, self.output_path_temp, self.file_name, False, error if error else f"不明なエラー (終了コード: {exitCode})")
        except Exception as e:
            if self._isRunning:
                self.finished.emit(self.file_url, self.output_path_temp, self.file_name, False, str(e))

    def stop(self):
        # スレッドの実行を停止するためのフラグを設定。
        self._isRunning = False

    def stop(self):
        # スレッドの実行を停止するためのフラグを設定。
        self._isRunning = False

    def stop(self):
        # スレッドの実行を停止するためのフラグを設定。
        self._isRunning = False


# バッチファイル処理
class BatchRunner(QThread):
    """バッチファイルを実行するスレッド (コマンドプロンプトウィンドウを閉じる)"""
    finished = pyqtSignal(str, int)

    def __init__(self, filepath):
        super().__init__()
        self.filepath = filepath

    def run(self):
        # バッチファイルを実行し、終了後にコマンドプロンプトウィンドウを閉じる
        command = f'cmd /c start cmd /k "{self.filepath} && exit"'
        process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
        stdout, stderr = process.communicate()
        exit_code = process.returncode
        output = stdout.decode('cp932', errors='ignore') + "\n" + stderr.decode('cp932', errors='ignore')
        self.finished.emit(self.filepath, exit_code)
        print(f"{self.filepath} の実行結果 (ウィンドウを閉じる):\n{output}")


# GeoJSON Viewer
class PlotWindow(QMainWindow):
    def __init__(self, geojson_data, parent=None):
        super().__init__(parent)
        self.setWindowTitle("GeoJSON Viewer")
        self.setGeometry(200, 200, 1200, 900)
        self.graphics_view = QGraphicsView()
        self.graphics_scene = QGraphicsScene()
        self.graphics_view.setScene(self.graphics_scene)
        self.setCentralWidget(self.graphics_view)
        self.min_geo_x = float('inf')
        self.max_geo_x = float('-inf')
        self.min_geo_y = float('inf')
        self.max_geo_y = float('-inf')
        self.plot_geojson_data(geojson_data)
        self.adjust_view()

    def update_bounds(self, x, y):
        self.min_geo_x = min(self.min_geo_x, x)
        self.max_geo_x = max(self.max_geo_x, x)
        self.min_geo_y = min(self.min_geo_y, y)
        self.max_geo_y = max(self.max_geo_y, y)

    def adjust_view(self):
        if self.graphics_scene.items() and self.min_geo_x != float('inf'):
            scene_rect = QRectF(self.min_geo_x, -self.max_geo_y,
                                self.max_geo_x - self.min_geo_x, self.max_geo_y - self.min_geo_y)
            self.graphics_scene.setSceneRect(scene_rect)
            self.graphics_view.fitInView(self.graphics_scene.sceneRect(), Qt.KeepAspectRatio)
            # スケールを2倍にする
            scale_factor = 2.0
            transform = QTransform().scale(scale_factor, scale_factor)
            self.graphics_view.setTransform(transform)

    def plot_geojson_data(self, geojson_data):
        self.graphics_scene.clear()
        self.min_geo_x = float('inf')
        self.max_geo_x = float('-inf')
        self.min_geo_y = float('inf')
        self.max_geo_y = float('-inf')
        if 'type' in geojson_data and geojson_data['type'] == 'FeatureCollection':
            for feature in geojson_data['features']:
                if 'geometry' in feature and feature['geometry'] and 'type' in feature['geometry'] and 'coordinates' in feature['geometry']:
                    geometry_type = feature['geometry']['type']
                    coordinates = feature['geometry']['coordinates']
                    self.draw_geometry(geometry_type, coordinates)
        elif 'type' in geojson_data and geojson_data['type'] == 'Feature' and 'geometry' in geojson_data and geojson_data['geometry']:
            geometry_type = geojson_data['geometry']['type']
            coordinates = geojson_data['geometry']['coordinates']
            self.draw_geometry(geometry_type, coordinates)
        else:
            QMessageBox.warning(self, "警告", "無効なGeoJSON形式です。")
        self.adjust_view()

    def draw_geometry(self, geometry_type, coordinates):
        fill_color = QColor(random.randint(0, 255), random.randint(0, 255), random.randint(0, 255), 150)
        stroke_color = Qt.black
        pen = QPen(stroke_color)
        brush = QBrush(fill_color)
        if geometry_type == 'Point':
            x, y = coordinates
            # 測量座標 -> 画面座標変換: XをXに、-YをYに
            draw_x, draw_y = x, -y
            radius = 5
            self.graphics_scene.addEllipse(draw_x - radius, draw_y - radius, 2 * radius, 2 * radius, pen, brush)
            self.update_bounds(x, y)
        elif geometry_type == 'LineString':
            points = []
            for x, y in coordinates:
                # 測量座標 -> 画面座標変換: XをXに、-YをYに
                draw_x, draw_y = x, -y
                points.append(QPointF(draw_x, draw_y))
                self.update_bounds(x, y)
            self.graphics_scene.addPolyline(points, pen)
        elif geometry_type == 'Polygon':
            for ring in coordinates:
                points = []
                for x, y in ring:
                    # 測量座標 -> 画面座標変換: XをXに、-YをYに
                    draw_x, draw_y = x, -y
                    points.append(QPointF(draw_x, draw_y))
                    self.update_bounds(x, y)
                polygon = QPolygonF(points)
                self.graphics_scene.addPolygon(polygon, pen, brush)
        elif geometry_type == 'MultiPoint':
            for x, y in coordinates:
                # 測量座標 -> 画面座標変換: XをXに、-YをYに
                draw_x, draw_y = x, -y
                radius = 5
                self.graphics_scene.addEllipse(draw_x - radius, draw_y - radius, 2 * radius, 2 * radius, pen, brush)
                self.update_bounds(x, y)
        elif geometry_type == 'MultiLineString':
            for line in coordinates:
                points = []
                for x, y in line:
                    # 測量座標 -> 画面座標変換: XをXに、-YをYに
                    draw_x, draw_y = x, -y
                    points.append(QPointF(draw_x, draw_y))
                    self.update_bounds(x, y)
                self.graphics_scene.addPolyline(points, pen)
        elif geometry_type == 'MultiPolygon':
            for polygon_coords in coordinates:
                for ring in polygon_coords:
                    points = []
                    for x, y in ring:
                        # 測量座標 -> 画面座標変換: XをXに、-YをYに
                        draw_x, draw_y = x, -y
                        points.append(QPointF(draw_x, draw_y))
                        self.update_bounds(x, y)
                    polygon = QPolygonF(points)
                    self.graphics_scene.addPolygon(polygon, pen, brush)


class Dialog(QMainWindow):
    def __init__(self):
        super().__init__()
        self.selected_folder = None
        self.initUI()
        self.geojson_files = []

    # G空間情報センターからファイルリストをダウンロードするアプリケーション。
    def __init__(self):
        super().__init__()
        self.initUI()
        # PowerShellコマンドの実行結果（ファイルリスト）を格納する変数
        self.downloaded_list = []
        # 都道府県フォルダ名リスト
        self.prefecture_folders = [
            "01_hokkaido", "02_aomori", "03_iwate", "04_miyagi", "05_akita",
            "06_yamagata", "07_fukushima", "08_ibaraki", "09_tochigi", "10_gumma",
            "11_saitama", "12_chiba", "13_tokyo", "14_kanagawa", "15_niigata",
            "16_toyama", "17_ishikawa", "18_fukui", "19_yamanashi", "20_nagano",
            "21_gifu", "22_shizuoka", "23_aichi", "24_mie", "25_shiga",
            "26_kyoto", "27_osaka", "28_hyogo", "29_nara", "30_wakayama",
            "31_tottori", "32_shimane", "33_okayama", "34_hiroshima", "35_yamaguchi",
            "36_tokushima", "37_kagawa", "38_ehime", "39_kochi", "40_fukuoka",
            "41_saga", "42_nagasaki", "43_kumamoto", "44_oita", "45_miyazaki",
            "46_kagoshima", "47_okinawa"
        ]
        self.max_concurrent_downloads = 5 # 同時ダウンロード数の制限
        self.active_downloads = 0
        self.download_queue = []
        self.downloaded_files = []
        self.total_downloads = 0
        self.completed_downloads = 0
        self.output_dir = ""
        self.workers = [] # 実行中のワーカーを管理するリスト
        self.download_in_progress = False # ダウンロード処理中フラグ
        self.move_mutex = QMutex() # ファイル移動処理の排他制御用ミューテックス

    def initUI(self):
#--------------------------------------------------------------------------------------------------------
        self.setObjectName("self")
        self.setEnabled(True)
        self.resize(525, 640)
        self.setMinimumSize(QtCore.QSize(525, 640))
        self.setMaximumSize(QtCore.QSize(525, 640))
        self.lineEdit_ogrmerge = QtWidgets.QLineEdit(self)
        self.lineEdit_ogrmerge.setGeometry(QtCore.QRect(20, 690, 351, 20))
        self.lineEdit_ogrmerge.setObjectName("lineEdit_ogrmerge")
        self.tabWidget = QtWidgets.QTabWidget(self)
        self.tabWidget.setGeometry(QtCore.QRect(0, 0, 531, 644))
        self.tabWidget.setObjectName("tabWidget")
        self.tab = QtWidgets.QWidget()
        self.tab.setObjectName("tab")
        self.groupBox = QtWidgets.QGroupBox(self.tab)
        self.groupBox.setGeometry(QtCore.QRect(11, 10, 501, 521))
        self.groupBox.setTitle("")
        self.groupBox.setObjectName("groupBox")
        self.unzip = QtWidgets.QPushButton(self.groupBox)
        self.unzip.setEnabled(True)
        self.unzip.setGeometry(QtCore.QRect(10, 30, 380, 25))
        self.unzip.setWhatsThis("")
        self.unzip.setAutoExclusive(False)
        self.unzip.setAutoDefault(True)
        self.unzip.setDefault(False)
        self.unzip.setObjectName("unzip")
        self.folder_label = QtWidgets.QLabel(self.groupBox)
        self.folder_label.setGeometry(QtCore.QRect(10, 6, 481, 20))
        self.folder_label.setFrameShape(QtWidgets.QFrame.StyledPanel)
        self.folder_label.setObjectName("folder_label")
        self.label_kaitou = QtWidgets.QLabel(self.groupBox)
        self.label_kaitou.setGeometry(QtCore.QRect(10, 62, 481, 21))
        self.label_kaitou.setFrameShape(QtWidgets.QFrame.StyledPanel)
        self.label_kaitou.setText("")
        self.label_kaitou.setObjectName("label_kaitou")
        self.counter = QtWidgets.QPushButton(self.groupBox)
        self.counter.setEnabled(True)
        self.counter.setGeometry(QtCore.QRect(10, 370, 481, 23))
        self.counter.setObjectName("counter")
        self.ikkatsu = QtWidgets.QCheckBox(self.groupBox)
        self.ikkatsu.setGeometry(QtCore.QRect(398, 33, 101, 20))
        self.ikkatsu.setChecked(False)
        self.ikkatsu.setObjectName("ikkatsu")
        self.create_batch = QtWidgets.QPushButton(self.groupBox)
        self.create_batch.setEnabled(True)
        self.create_batch.setGeometry(QtCore.QRect(10, 90, 380, 23))
        self.create_batch.setObjectName("create_batch")
        self.generate_button = QtWidgets.QPushButton(self.groupBox)
        self.generate_button.setEnabled(True)
        self.generate_button.setGeometry(QtCore.QRect(10, 120, 380, 23))
        self.generate_button.setObjectName("generate_button")
        self.run_button = QtWidgets.QPushButton(self.groupBox)
        self.run_button.setEnabled(True)
        self.run_button.setGeometry(QtCore.QRect(10, 150, 311, 23))
        self.run_button.setObjectName("run_button")
        self.format1 = QtWidgets.QRadioButton(self.groupBox)
        self.format1.setGeometry(QtCore.QRect(330, 154, 81, 16))
        self.format1.setChecked(True)
        self.format1.setObjectName("format1")
        self.format2 = QtWidgets.QRadioButton(self.groupBox)
        self.format2.setGeometry(QtCore.QRect(408, 154, 81, 16))
        self.format2.setObjectName("format2")
        self.label_2 = QtWidgets.QLabel(self.groupBox)
        self.label_2.setGeometry(QtCore.QRect(10, 180, 221, 16))
        self.label_2.setObjectName("label_2")
        self.batch_list = QtWidgets.QListWidget(self.groupBox)
        self.batch_list.setGeometry(QtCore.QRect(10, 200, 481, 71))
        self.batch_list.setObjectName("batch_list")
        self.output_list = QtWidgets.QListWidget(self.groupBox)
        self.output_list.setGeometry(QtCore.QRect(10, 420, 481, 91))
        self.output_list.setObjectName("output_list")
        self.label_3 = QtWidgets.QLabel(self.groupBox)
        self.label_3.setGeometry(QtCore.QRect(10, 400, 221, 16))
        self.label_3.setObjectName("label_3")
        self.ikkatsu2 = QtWidgets.QCheckBox(self.groupBox)
        self.ikkatsu2.setGeometry(QtCore.QRect(398, 92, 91, 20))
        self.ikkatsu2.setChecked(False)
        self.ikkatsu2.setObjectName("ikkatsu2")
        self.ikkatsu3 = QtWidgets.QCheckBox(self.groupBox)
        self.ikkatsu3.setGeometry(QtCore.QRect(398, 122, 91, 20))
        self.ikkatsu3.setChecked(False)
        self.ikkatsu3.setObjectName("ikkatsu3")
        self.pmtiles = QtWidgets.QPushButton(self.groupBox)
        self.pmtiles.setEnabled(True)
        self.pmtiles.setGeometry(QtCore.QRect(180, 280, 210, 23))
        self.pmtiles.setObjectName("pmtiles")
        self.progress_bar = QtWidgets.QProgressBar(self.groupBox)
        self.progress_bar.setGeometry(QtCore.QRect(10, 340, 481, 23))
        self.progress_bar.setProperty("value", 0)
        self.progress_bar.setObjectName("progress_bar")
        self.process_xml_files = QtWidgets.QPushButton(self.groupBox)
        self.process_xml_files.setEnabled(True)
        self.process_xml_files.setGeometry(QtCore.QRect(10, 310, 481, 23))
        self.process_xml_files.setObjectName("process_xml_files")
        self.label_8 = QtWidgets.QLabel(self.groupBox)
        self.label_8.setGeometry(QtCore.QRect(10, 285, 81, 16))
        self.label_8.setObjectName("label_8")
        self.ubuntu = QtWidgets.QLineEdit(self.groupBox)
        self.ubuntu.setGeometry(QtCore.QRect(94, 282, 81, 20))
        self.ubuntu.setAlignment(QtCore.Qt.AlignCenter)
        self.ubuntu.setObjectName("ubuntu")
        self.ikkatsu4 = QtWidgets.QCheckBox(self.groupBox)
        self.ikkatsu4.setGeometry(QtCore.QRect(398, 281, 101, 20))
        self.ikkatsu4.setChecked(False)
        self.ikkatsu4.setObjectName("ikkatsu4")
        self.groupBox_2 = QtWidgets.QGroupBox(self.tab)
        self.groupBox_2.setGeometry(QtCore.QRect(11, 540, 501, 71))
        self.groupBox_2.setObjectName("groupBox_2")
        self.PythonPath = QtWidgets.QPushButton(self.groupBox_2)
        self.PythonPath.setEnabled(True)
        self.PythonPath.setGeometry(QtCore.QRect(390, 16, 101, 23))
        self.PythonPath.setObjectName("PythonPath")
        self.mojxml2geojsonPath = QtWidgets.QPushButton(self.groupBox_2)
        self.mojxml2geojsonPath.setEnabled(True)
        self.mojxml2geojsonPath.setGeometry(QtCore.QRect(390, 40, 101, 23))
        self.mojxml2geojsonPath.setObjectName("mojxml2geojsonPath")
        self.lineEdit_python = QtWidgets.QLineEdit(self.groupBox_2)
        self.lineEdit_python.setGeometry(QtCore.QRect(8, 18, 371, 20))
        self.lineEdit_python.setObjectName("lineEdit_python")
        self.lineEdit_mojxml2geojson = QtWidgets.QLineEdit(self.groupBox_2)
        self.lineEdit_mojxml2geojson.setGeometry(QtCore.QRect(8, 42, 371, 20))
        self.lineEdit_mojxml2geojson.setObjectName("lineEdit_mojxml2geojson")
        self.tabWidget.addTab(self.tab, "")
        self.tab_2 = QtWidgets.QWidget()
        self.tab_2.setObjectName("tab_2")
        self.groupBox_3 = QtWidgets.QGroupBox(self.tab_2)
        self.groupBox_3.setGeometry(QtCore.QRect(11, 10, 501, 601))
        self.groupBox_3.setMinimumSize(QtCore.QSize(501, 601))
        self.groupBox_3.setMaximumSize(QtCore.QSize(501, 601))
        self.groupBox_3.setObjectName("groupBox_3")
        self.groupBox_4 = QtWidgets.QGroupBox(self.groupBox_3)
        self.groupBox_4.setGeometry(QtCore.QRect(5, 40, 491, 181))
        self.groupBox_4.setTitle("")
        self.groupBox_4.setObjectName("groupBox_4")

        # 固定 書き換え禁止！！！----------------------------------------------------------------
        layout = QtWidgets.QGridLayout(self.groupBox_4) # GridLayoutを使用
        self.checkboxes = []
        self.checkboxes_dict = {} # objectNameでチェックボックスを検索できるようにするための辞書
        row = 0
        col = 0
        for i in range(1, 48):
            checkbox_name = f"c_{i:02d}"
            checkbox = QtWidgets.QCheckBox(self.groupBox_4)
            checkbox.setObjectName(checkbox_name)
            checkbox.stateChanged.connect(self.checkbox_state_changed)
            self.checkboxes.append(checkbox)
            self.checkboxes_dict[checkbox_name] = checkbox
            layout.addWidget(checkbox, row, col) # レイアウトに追加
            col += 1
            if col > 5: # 例として6列ごとに改行
                col = 0
                row += 1
        # 固定 書き換え禁止！！！----------------------------------------------------------------

        self.label = QtWidgets.QLabel(self.groupBox_3)
        self.label.setGeometry(QtCore.QRect(10, 16, 161, 16))
        self.label.setObjectName("label")
        self.apikey_edit = QtWidgets.QLineEdit(self.groupBox_3)
        self.apikey_edit.setGeometry(QtCore.QRect(170, 13, 141, 20))
        self.apikey_edit.setInputMask("")
        self.apikey_edit.setText("")
        self.apikey_edit.setObjectName("apikey_edit")
        self.label_4 = QtWidgets.QLabel(self.groupBox_3)
        self.label_4.setGeometry(QtCore.QRect(330, 16, 91, 16))
        self.label_4.setObjectName("label_4")
        self.year = QtWidgets.QLineEdit(self.groupBox_3)
        self.year.setGeometry(QtCore.QRect(422, 13, 71, 20))
        self.year.setAlignment(QtCore.Qt.AlignCenter)
        self.year.setObjectName("year")
        self.file_list_widget = QtWidgets.QListWidget(self.groupBox_3)
        self.file_list_widget.setGeometry(QtCore.QRect(10, 331, 371, 241))
        self.file_list_widget.setSelectionMode(QtWidgets.QAbstractItemView.ExtendedSelection)
        self.file_list_widget.setObjectName("file_list_widget")
        self.select_all = QtWidgets.QPushButton(self.groupBox_3)
        self.select_all.setEnabled(True)
        self.select_all.setGeometry(QtCore.QRect(390, 410, 101, 25))
        self.select_all.setWhatsThis("")
        self.select_all.setAutoExclusive(False)
        self.select_all.setAutoDefault(True)
        self.select_all.setDefault(False)
        self.select_all.setObjectName("select_all")
        self.deselect_all = QtWidgets.QPushButton(self.groupBox_3)
        self.deselect_all.setEnabled(True)
        self.deselect_all.setGeometry(QtCore.QRect(390, 440, 101, 25))
        self.deselect_all.setWhatsThis("")
        self.deselect_all.setAutoExclusive(False)
        self.deselect_all.setAutoDefault(True)
        self.deselect_all.setDefault(False)
        self.deselect_all.setObjectName("deselect_all")
        self.download_selected_button = QtWidgets.QPushButton(self.groupBox_3)
        self.download_selected_button.setEnabled(True)
        self.download_selected_button.setGeometry(QtCore.QRect(390, 380, 101, 25))
        self.download_selected_button.setWhatsThis("")
        self.download_selected_button.setAutoExclusive(False)
        self.download_selected_button.setAutoDefault(True)
        self.download_selected_button.setDefault(False)
        self.download_selected_button.setObjectName("download_selected_button")
        self.download_list_button = QtWidgets.QPushButton(self.groupBox_3)
        self.download_list_button.setEnabled(True)
        self.download_list_button.setGeometry(QtCore.QRect(390, 330, 101, 25))
        self.download_list_button.setWhatsThis("")
        self.download_list_button.setAutoExclusive(False)
        self.download_list_button.setAutoDefault(True)
        self.download_list_button.setDefault(False)
        self.download_list_button.setObjectName("download_list_button")
        self.download_status_label = QtWidgets.QLabel(self.groupBox_3)
        self.download_status_label.setGeometry(QtCore.QRect(20, 578, 91, 16))
        self.download_status_label.setObjectName("download_status_label")
        self.download_status_text = QtWidgets.QLabel(self.groupBox_3)
        self.download_status_text.setGeometry(QtCore.QRect(120, 578, 381, 16))
        self.download_status_text.setText("")
        self.download_status_text.setObjectName("download_status_text")
        self.ikkatsu0 = QtWidgets.QCheckBox(self.groupBox_3)
        self.ikkatsu0.setGeometry(QtCore.QRect(392, 358, 111, 20))
        self.ikkatsu0.setChecked(False)
        self.ikkatsu0.setObjectName("ikkatsu0")
        self.output_list2 = QtWidgets.QListWidget(self.groupBox_3)
        self.output_list2.setGeometry(QtCore.QRect(10, 240, 481, 81))
        self.output_list2.setObjectName("output_list2")
        self.label_5 = QtWidgets.QLabel(self.groupBox_3)
        self.label_5.setGeometry(QtCore.QRect(10, 222, 221, 16))
        self.label_5.setObjectName("label_5")
        self.kensaku_text = QtWidgets.QLineEdit(self.groupBox_3)
        self.kensaku_text.setGeometry(QtCore.QRect(390, 470, 101, 20))
        self.kensaku_text.setObjectName("kensaku_text")
        self.kensaku = QtWidgets.QPushButton(self.groupBox_3)
        self.kensaku.setGeometry(QtCore.QRect(390, 494, 101, 23))
        self.kensaku.setObjectName("kensaku")
        self.tabWidget.addTab(self.tab_2, "")
        self.tab_3 = QtWidgets.QWidget()
        self.tab_3.setObjectName("tab_3")
        self.geojson_folder = QtWidgets.QPushButton(self.tab_3)
        self.geojson_folder.setGeometry(QtCore.QRect(10, 10, 121, 23))
        self.geojson_folder.setObjectName("geojson_folder")
        self.convert_button = QtWidgets.QPushButton(self.tab_3)
        self.convert_button.setGeometry(QtCore.QRect(137, 10, 121, 23))
        self.convert_button.setObjectName("convert_button")
        self.geojson_list = QtWidgets.QListWidget(self.tab_3)
        self.geojson_list.setGeometry(QtCore.QRect(10, 62, 501, 441))
        self.geojson_list.setObjectName("geojson_list")
        self.output_list3 = QtWidgets.QListWidget(self.tab_3)
        self.output_list3.setGeometry(QtCore.QRect(10, 528, 501, 81))
        self.output_list3.setObjectName("output_list3")
        self.label_6 = QtWidgets.QLabel(self.tab_3)
        self.label_6.setGeometry(QtCore.QRect(10, 507, 221, 16))
        self.label_6.setObjectName("label_6")
        self.mapping = QtWidgets.QPushButton(self.tab_3)
        self.mapping.setGeometry(QtCore.QRect(390, 10, 121, 23))
        self.mapping.setObjectName("mapping")
        self.convert_button2 = QtWidgets.QPushButton(self.tab_3)
        self.convert_button2.setGeometry(QtCore.QRect(264, 10, 121, 23))
        self.convert_button2.setObjectName("convert_button2")
        self.label_7 = QtWidgets.QLabel(self.tab_3)
        self.label_7.setGeometry(QtCore.QRect(10, 40, 121, 16))
        self.label_7.setObjectName("label_7")
        self.scale_edit = QtWidgets.QLineEdit(self.tab_3)
        self.scale_edit.setGeometry(QtCore.QRect(137, 38, 121, 20))
        self.scale_edit.setAlignment(QtCore.Qt.AlignCenter)
        self.scale_edit.setObjectName("scale_edit")
        self.convert_button3 = QtWidgets.QPushButton(self.tab_3)
        self.convert_button3.setGeometry(QtCore.QRect(264, 36, 121, 23))
        self.convert_button3.setObjectName("convert_button3")
        self.kei = QtWidgets.QComboBox(self.tab_3)
        self.kei.setGeometry(QtCore.QRect(448, 36, 41, 22))
        self.kei.setLayoutDirection(QtCore.Qt.LeftToRight)
        self.kei.setObjectName("kei")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.kei.addItem("")
        self.label_9 = QtWidgets.QLabel(self.tab_3)
        self.label_9.setGeometry(QtCore.QRect(392, 40, 61, 16))
        self.label_9.setObjectName("label_9")
        self.label_10 = QtWidgets.QLabel(self.tab_3)
        self.label_10.setGeometry(QtCore.QRect(496, 40, 21, 16))
        self.label_10.setObjectName("label_10")
        self.label_9.raise_()
        self.geojson_folder.raise_()
        self.convert_button.raise_()
        self.geojson_list.raise_()
        self.output_list3.raise_()
        self.label_6.raise_()
        self.mapping.raise_()
        self.convert_button2.raise_()
        self.label_7.raise_()
        self.scale_edit.raise_()
        self.convert_button3.raise_()
        self.kei.raise_()
        self.label_10.raise_()
        self.tabWidget.addTab(self.tab_3, "")
        self.tab_4 = QtWidgets.QWidget()
        self.tab_4.setObjectName("tab_4")
        self.sima_text = QtWidgets.QTextEdit(self.tab_4)
        self.sima_text.setGeometry(QtCore.QRect(10, 40, 501, 571))
        self.sima_text.setObjectName("sima_text")
        self.xml = QtWidgets.QPushButton(self.tab_4)
        self.xml.setGeometry(QtCore.QRect(10, 10, 91, 23))
        self.xml.setObjectName("xml")
        self.check_shubetsu = QtWidgets.QCheckBox(self.tab_4)
        self.check_shubetsu.setGeometry(QtCore.QRect(120, 13, 201, 16))
        self.check_shubetsu.setObjectName("check_shubetsu")
        self.tabWidget.addTab(self.tab_4, "")

        self.tabWidget.setCurrentIndex(0)
        QtCore.QMetaObject.connectSlotsByName(self)

        _translate = QtCore.QCoreApplication.translate
        self.setWindowTitle(_translate("self", "ConvertTool for Python"))
        self.unzip.setText(_translate("self", "1. ZIPファイル解凍処理"))
        self.folder_label.setText(_translate("self", "選択中のフォルダ : "))
        self.counter.setText(_translate("self", "ファイル数確認 "))
        self.ikkatsu.setText(_translate("self", "1,2,3 連続処理"))
        self.create_batch.setText(_translate("self", "2. GeoJSON作成用バッチファイル作成 （mojxml2geojson）"))
        self.generate_button.setText(_translate("self", "3. FlatGeobuf作成用バッチファイル作成 （ogrmerge）"))
        self.run_button.setText(_translate("self", "4. バッチファイル実行（並列処理）"))
        self.format1.setText(_translate("self", "GeoJSON"))
        self.format2.setText(_translate("self", "FlatGeobuf"))
        self.label_2.setText(_translate("self", "検出されたバッチファイル"))
        self.label_3.setText(_translate("self", "実行ログ"))
        self.ikkatsu2.setText(_translate("self", "2,4連続処理"))
        self.ikkatsu3.setText(_translate("self", "3,4連続処理"))
        self.pmtiles.setText(_translate("self", "5. PMTiles作成"))
        self.process_xml_files.setText(_translate("self", "XMLファイル分別処理"))
        self.label_8.setText(_translate("self", "Ubuntu version"))
        self.ubuntu.setText(_translate("self", "Ubuntu-22.04"))
        self.ikkatsu4.setText(_translate("self", "3,4,5 連続処理"))
        self.groupBox_2.setTitle(_translate("self", "Path 設定 （mojxml2geojson は K\'z lab 改造版）"))
        self.PythonPath.setText(_translate("self", "Python"))
        self.mojxml2geojsonPath.setText(_translate("self", "mojxml2geojson"))
        self.tabWidget.setTabText(self.tabWidget.indexOf(self.tab), _translate("self", "CONVERT"))
        self.groupBox_3.setTitle(_translate("self", "0. 地図XMLのダウンロード"))

        # 固定 書き換え禁止！！！----------------------------------------------------------------
        for name, checkbox in self.checkboxes_dict.items():
            if name == "c_01":
                checkbox.setText(_translate("self", "北海道"))
            elif name == "c_02":
                checkbox.setText(_translate("self", "青森県"))
            elif name == "c_03":
                checkbox.setText(_translate("self", "岩手県"))
            elif name == "c_04":
                checkbox.setText(_translate("self", "宮城県"))
            elif name == "c_05":
                checkbox.setText(_translate("self", "秋田県"))
            elif name == "c_06":
                checkbox.setText(_translate("self", "山形県"))
            elif name == "c_07":
                checkbox.setText(_translate("self", "福島県"))
            elif name == "c_08":
                checkbox.setText(_translate("self", "茨城県"))
            elif name == "c_09":
                checkbox.setText(_translate("self", "栃木県"))
            elif name == "c_10":
                checkbox.setText(_translate("self", "群馬県"))
            elif name == "c_11":
                checkbox.setText(_translate("self", "埼玉県"))
            elif name == "c_12":
                checkbox.setText(_translate("self", "千葉県"))
            elif name == "c_13":
                checkbox.setText(_translate("self", "東京都"))
            elif name == "c_14":
                checkbox.setText(_translate("self", "神奈川県"))
            elif name == "c_15":
                checkbox.setText(_translate("self", "新潟県"))
            elif name == "c_16":
                checkbox.setText(_translate("self", "富山県"))
            elif name == "c_17":
                checkbox.setText(_translate("self", "石川県"))
            elif name == "c_18":
                checkbox.setText(_translate("self", "福井県"))
            elif name == "c_19":
                checkbox.setText(_translate("self", "山梨県"))
            elif name == "c_20":
                checkbox.setText(_translate("self", "長野県"))
            elif name == "c_21":
                checkbox.setText(_translate("self", "岐阜県"))
            elif name == "c_22":
                checkbox.setText(_translate("self", "静岡県"))
            elif name == "c_23":
                checkbox.setText(_translate("self", "愛知県"))
            elif name == "c_24":
                checkbox.setText(_translate("self", "三重県"))
            elif name == "c_25":
                checkbox.setText(_translate("self", "滋賀県"))
            elif name == "c_26":
                checkbox.setText(_translate("self", "京都府"))
            elif name == "c_27":
                checkbox.setText(_translate("self", "大阪府"))
            elif name == "c_28":
                checkbox.setText(_translate("self", "兵庫県"))
            elif name == "c_29":
                checkbox.setText(_translate("self", "奈良県"))
            elif name == "c_30":
                checkbox.setText(_translate("self", "和歌山県"))
            elif name == "c_31":
                checkbox.setText(_translate("self", "鳥取県"))
            elif name == "c_32":
                checkbox.setText(_translate("self", "島根県"))
            elif name == "c_33":
                checkbox.setText(_translate("self", "岡山県"))
            elif name == "c_34":
                checkbox.setText(_translate("self", "広島県"))
            elif name == "c_35":
                checkbox.setText(_translate("self", "山口県"))
            elif name == "c_36":
                checkbox.setText(_translate("self", "徳島県"))
            elif name == "c_37":
                checkbox.setText(_translate("self", "香川県"))
            elif name == "c_38":
                checkbox.setText(_translate("self", "愛媛県"))
            elif name == "c_39":
                checkbox.setText(_translate("self", "高知県"))
            elif name == "c_40":
                checkbox.setText(_translate("self", "福岡県"))
            elif name == "c_41":
                checkbox.setText(_translate("self", "佐賀県"))
            elif name == "c_42":
                checkbox.setText(_translate("self", "長崎県"))
            elif name == "c_43":
                checkbox.setText(_translate("self", "熊本県"))
            elif name == "c_44":
                checkbox.setText(_translate("self", "大分県"))
            elif name == "c_45":
                checkbox.setText(_translate("self", "宮崎県"))
            elif name == "c_46":
                checkbox.setText(_translate("self", "鹿児島県"))
            elif name == "c_47":
                checkbox.setText(_translate("self", "沖縄県"))
        # 固定 書き換え禁止！！！----------------------------------------------------------------

        self.label.setText(_translate("self", "G空間情報センターの API Key"))
        self.label_4.setText(_translate("self", "地図XMLの年度"))
        self.year.setText(_translate("self", "2025"))
        self.select_all.setText(_translate("self", "全選択"))
        self.deselect_all.setText(_translate("self", "前解除"))
        self.download_selected_button.setText(_translate("self", "ダウンロード"))
        self.download_list_button.setText(_translate("self", "リスト取得"))
        self.download_status_label.setText(_translate("self", "ダウンロード状況 ： "))
        self.ikkatsu0.setText(_translate("self", "0,1 連続処理"))
        self.label_5.setText(_translate("self", "エラーログ"))
        self.kensaku.setText(_translate("self", "検索"))
        self.tabWidget.setTabText(self.tabWidget.indexOf(self.tab_2), _translate("self", "DOWNLOAD"))
        self.geojson_folder.setText(_translate("self", "フォルダ選択"))
        self.convert_button.setText(_translate("self", "SIMA 変換"))
        self.label_6.setText(_translate("self", "実行ログ"))
        self.mapping.setText(_translate("self", "描画"))
        self.convert_button2.setText(_translate("self", "DXF/SFC 変換"))
        self.label_7.setText(_translate("self", "DXオプション  拡大率 ："))
        self.scale_edit.setText(_translate("self", "39.3701594308"))
        self.convert_button3.setText(_translate("self", "KML 変換"))
        self.kei.setItemText(0, _translate("self", "1"))
        self.kei.setItemText(1, _translate("self", "2"))
        self.kei.setItemText(2, _translate("self", "3"))
        self.kei.setItemText(3, _translate("self", "4"))
        self.kei.setItemText(4, _translate("self", "5"))
        self.kei.setItemText(5, _translate("self", "6"))
        self.kei.setItemText(6, _translate("self", "7"))
        self.kei.setItemText(7, _translate("self", "8"))
        self.kei.setItemText(8, _translate("self", "9"))
        self.kei.setItemText(9, _translate("self", "10"))
        self.kei.setItemText(10, _translate("self", "11"))
        self.kei.setItemText(11, _translate("self", "12"))
        self.kei.setItemText(12, _translate("self", "13"))
        self.kei.setItemText(13, _translate("self", "14"))
        self.kei.setItemText(14, _translate("self", "15"))
        self.kei.setItemText(15, _translate("self", "16"))
        self.kei.setItemText(16, _translate("self", "17"))
        self.kei.setItemText(17, _translate("self", "18"))
        self.kei.setItemText(18, _translate("self", "13"))
        self.label_9.setText(_translate("self", "公共座標"))
        self.label_10.setText(_translate("self", "系"))
        self.tabWidget.setTabText(self.tabWidget.indexOf(self.tab_3), _translate("self", "CONVERT2"))
        self.xml.setText(_translate("self", "地図XML"))
        self.check_shubetsu.setText(_translate("self", "点名の先頭に基準点種別を付ける"))
        self.tabWidget.setTabText(self.tabWidget.indexOf(self.tab_4), _translate("self", "基準点抽出"))

#--------------------------------------------------------------------------------------------------------

        self.apikey_edit.setEchoMode(QLineEdit.Password) # API Key を伏せ文字にする

        self.show()

        self.unzip.clicked.connect(self.kaitou)
        self.process_xml_files.clicked.connect(self.processXmlFiles)
        self.counter.clicked.connect(self.countFiles)
        self.create_batch.clicked.connect(self.createPublicCoordinateBatch2)
        self.generate_button.clicked.connect(self.generate_batch_files2)
        self.PythonPath.clicked.connect(self.set_path_python)
        self.mojxml2geojsonPath.clicked.connect(self.set_path_mojxml2geojson)
        self.run_button.clicked.connect(self.run_selected_batches)
        self.pmtiles.clicked.connect(self.convert_pmtiles)
        self.download_list_button.clicked.connect(self.executeGetFileList)
        self.download_selected_button.clicked.connect(self.startDownload)
        self.select_all.clicked.connect(self.select_all_items)
        self.deselect_all.clicked.connect(self.deselect_all_items)
        self.geojson_folder.clicked.connect(self.select_geojson_folder)
        self.geojson_list.setSelectionMode(QAbstractItemView.SingleSelection)
        self.convert_button.clicked.connect(self.process_selected_geojson)
        self.convert_button2.clicked.connect(self.convert_selected_geojson)
        self.mapping.clicked.connect(self.draw_selected_geojson)
        self.geojson_files = []
        self.selected_geojson_data = None
        self.current_plot_window = None
        self.kensaku.clicked.connect(self.search_and_select)
        self.ikkatsu.clicked.connect(self.SetCheckBox)
        self.ikkatsu3.clicked.connect(self.SetCheckBox2)
        self.ikkatsu4.clicked.connect(self.SetCheckBox3)
        self.convert_button3.clicked.connect(self.convert_to_kml)
        self.kei.currentIndexChanged.connect(self.savesetting)
        self.xml.clicked.connect(self.open_xml)

        # 設定復元
        settingfname = os.path.dirname(__file__) + u"/convert_tool.ini"
        if os.path.exists(settingfname):
            try:
                for line in codecs.open(settingfname,"r", "utf-8"):
                    dat = line.replace("\n","").split("=")
                    if "python_path" in line:
                        self.lineEdit_python.setText(dat[1])
                    elif "ogrmerge_path" in line:
                        self.lineEdit_ogrmerge.setText(dat[1])
                    elif "mojxml2geojson_path" in line:
                        self.lineEdit_mojxml2geojson.setText(dat[1])
                    elif "API_Key" in line:
                        if dat[1].strip() == "":
                            self.apikey_edit.setText("")
                        else:
                            self.apikey_edit.setText(dat[1])
                    elif "Year" in line:
                        if dat[1].strip() == "":
                            self.year.setText("2025")
                            self.savesetting()
                        else:
                            self.year.setText(dat[1])
                    elif "Ubuntu" in line:
                        if dat[1].strip() == "":
                            self.ubuntu.setText("Ubuntu-22.04")
                            self.savesetting()
                        else:
                            self.ubuntu.setText(dat[1])
                    elif "Kei" in line:
                        if dat[1].strip() == "":
                            self.kei.setCurrentIndex(12)  # デフォルトは13系
                            self.savesetting()
                        else:
                            self.kei.setCurrentIndex(int(dat[1]))
                    elif "Scale" in line:
                        if dat[1].strip() == "":
                            self.scale_edit.setText("39.3701594308")
                            self.savesetting()
                        else:
                            self.scale_edit.setText(dat[1])
            except:
                pass
        else:
            setting_text = "python_path=" + "\n"
            setting_text = setting_text + "ogrmerge_path=" + "\n"
            setting_text = setting_text + "mojxml2geojson_path=" + "\n"
            setting_text = setting_text + "API_Key=" + "\n"
            setting_text = setting_text + "Year=" + "\n"
            setting_text = setting_text + "Ubuntu=" + "\n"
            setting_text = setting_text + "Kei=" + "\n"
            setting_text = setting_text + "Scale=" + "\n"
            with open(settingfname, "w", encoding="utf-8") as f:
                 f.write(setting_text)

        # 起動時に xml_list.txt からファイルリストを読み込む。
        file_path = os.path.dirname(__file__) + "/xml_list.txt"
        try:
            with open(file_path, "r", encoding="utf-8") as f:
                for i, line in enumerate(f):
                    url = line.strip()
                    if i == 0 and url.startswith('\ufeff'):
                        url = url[1:] # 先頭のBOMを削除
                    if url:
                        self.file_list_widget.addItem(url)
            if self.file_list_widget.count() > 0:
                self.download_selected_button.setEnabled(True)
        except FileNotFoundError:
            QMessageBox.warning(self, "警告", f"ファイル '{file_path}' が見つかりませんでした。")
        except Exception as e:
            QMessageBox.critical(self, "エラー", f"ファイル '{file_path}' の読み込み中にエラーが発生しました: {e}")

        # 検索用リストをメモリに読み込む
        self.read_csv_data()

    def cleanup_threads(self):
        # アプリケーション終了前にすべてのスレッドが終了するのを待つ。
        for worker in self.workers:
            if worker.isRunning():
                worker.stop()
                worker.wait()

    # チェックボックスセット
    def SetCheckBox(self):
        if self.ikkatsu.isChecked():
            self.ikkatsu2.setChecked(True)
            self.ikkatsu3.setChecked(True)
            self.ikkatsu4.setChecked(True)

    # チェックボックスセット
    def SetCheckBox2(self):
        if self.ikkatsu3.isChecked():
            self.ikkatsu4.setChecked(True)

    # チェックボックスセット
    def SetCheckBox3(self):
        if self.ikkatsu4.isChecked():
            self.ikkatsu3.setChecked(True)

    # Pythonフォルダ選択
    def set_path_python(self):
        if self.lineEdit_python.text() != "":
            python_folder = str(self.lineEdit_python.text().replace("/python.exe", "").replace("\r", ""))
            ogrmerge_path = python_folder + "/Scripts/ogrmerge.py"
            self.lineEdit_ogrmerge.setText("ogrmerge_path=" + ogrmerge_path)
        self.savesetting()
        folder_name = QFileDialog.getExistingDirectory(self, "python.exeのあるフォルダを選択")
        if folder_name:
            self.lineEdit_python.setText(folder_name + "/python.exe")
            self.savesetting()

    # mojxml2geojson（K'z lab改造版）フォルダ選択
    def set_path_mojxml2geojson(self):
        self.savesetting()
        folder_name = QFileDialog.getExistingDirectory(self, "mojxml2geojson（K'z lab改造版）のフォルダを選択")
        if folder_name:
            self.lineEdit_mojxml2geojson.setText(folder_name)
            self.savesetting()

    # 設定保存
    def savesetting(self):
        settingfname = os.path.dirname(__file__) + u"/convert_tool.ini"
        setting_text = "python_path=" + str(self.lineEdit_python.text().replace("\r", "")) +"\n"
        setting_text = setting_text + "ogrmerge_path=" + str(self.lineEdit_ogrmerge.text().replace("\r", "")) +"\n"
        setting_text = setting_text + "mojxml2geojson_path=" + str(self.lineEdit_mojxml2geojson.text().replace("\r", "")) +"\n"
        setting_text = setting_text + "API_Key=" + str(self.apikey_edit.text().replace("\r", "")) +"\n"
        setting_text = setting_text + "Year=" + str(self.year.text().replace("\r", "")) +"\n"
        setting_text = setting_text + "Ubuntu=" + str(self.ubuntu.text().replace("\r", "")) +"\n"
        setting_text = setting_text + "Kei=" + str(self.kei.currentIndex()) +"\n"
        setting_text = setting_text + "Scale=" + str(self.scale_edit.text().replace("\r", "")) +"\n"
        with open(settingfname, "w", encoding="utf-8") as f:
             f.write(setting_text)

    # 1. 解凍処理 ～進捗状況～
    def update_label_kaitou(self, text):
        self.label_kaitou.setText(text)
        QApplication.processEvents() # GUIを更新

    # 1. 解凍処理
    def kaitou(self):
        if self.ikkatsu0.isChecked():
            folder_name = self.folder_label.text().replace("選択中のフォルダ : ", "")
        else:
            folder_name = QFileDialog.getExistingDirectory(self, "フォルダを選択")
        if folder_name:
            self.folder_label.setText(f'選択中のフォルダ : {folder_name}')
            root_directory = folder_name  # 親フォルダを選択
            self.process_folders(root_directory) # self (Dialogのインスタンス) を渡す
            self.update_label_kaitou("全てのフォルダの処理が完了しました。")
            if self.ikkatsu.isChecked():
                self.createPublicCoordinateBatch()

    # 1. 解凍処理
    def process_folders(self, root_dir):
        folder_names = [item for item in os.listdir(root_dir) if os.path.isdir(os.path.join(root_dir, item))]
        total_folders = len(folder_names)
        for i, folder_name in enumerate(folder_names):
            message = f"処理中フォルダ: {folder_name} ({i+1}/{total_folders})"
            self.update_label_kaitou(message)
            self.process_folder(os.path.join(root_dir, folder_name))

    # 1. 解凍処理
    def process_folder(self, folder_path):
        base_folder_name = os.path.basename(folder_path)
        self.update_label_kaitou(f"  - {base_folder_name} 内の処理を開始...")
        zip2_path = os.path.join(folder_path, "zip2")
        xml_path = os.path.join(folder_path, "xml")
        extracted_zip_count = 0
        extracted_csv_count = 0
        has_outer_zip = False
        zip_files_in_folder = [f for f in os.listdir(folder_path) if f.endswith(".zip")]
        total_zip_files = len(zip_files_in_folder)
        for i, filename in enumerate(zip_files_in_folder):
            has_outer_zip = True
            zip_file_path = os.path.join(folder_path, filename)
            self.update_label_kaitou(f"    - 一次処理中 ZIPファイル: {filename} ({i+1}/{total_zip_files})")
            try:
                with zipfile.ZipFile(zip_file_path, 'r') as outer_zip:
                    # zip2フォルダがなければ作成
                    os.makedirs(zip2_path, exist_ok=True)
                    # xmlフォルダがなければ作成
                    os.makedirs(xml_path, exist_ok=True)
                    inner_zip_files = [name for name in outer_zip.namelist() if name.endswith(".zip")]
                    total_inner_zip_files = len(inner_zip_files)
                    inner_csv_files_direct = [name for name in outer_zip.namelist() if name.endswith(".csv")]
                    total_inner_csv_files_direct = len(inner_csv_files_direct)
                    for j, inner_filename in enumerate(inner_zip_files):
                        self.update_label_kaitou(f"      - 二次処理中 ZIPファイル: {inner_filename} ({j+1}/{total_inner_zip_files})")
                        outer_zip.extract(inner_filename, zip2_path)
                        extracted_zip_count += 1
                        inner_zip_path = os.path.join(zip2_path, inner_filename)
                        try:
                            with zipfile.ZipFile(inner_zip_path, 'r') as inner_zip:
                                extracted_files = [name for name in inner_zip.namelist() if name.endswith(".xml") or name.endswith(".csv")]
                                total_extracted_files = len(extracted_files)
                                for k, extracted_filename in enumerate(extracted_files):
                                    self.update_label_kaitou(f"        - 解凍中ファイル: {extracted_filename} ({k+1}/{total_extracted_files})")
                                    inner_zip.extract(extracted_filename, xml_path)
                                    if extracted_filename.endswith(".csv"):
                                        extracted_csv_count += 1
                        except zipfile.BadZipFile:
                            self.update_label_kaitou(f"      警告: 二次解凍に失敗しました - {inner_zip_path}")
            except zipfile.BadZipFile:
                self.update_label_kaitou(f"    警告: 一次解凍に失敗しました - {zip_file_path}")
        # zipファイルが存在し、二次解凍処理が完了したらzip2フォルダを削除
        if has_outer_zip and os.path.exists(zip2_path):
            pass
            try:
                shutil.rmtree(zip2_path)
                self.update_label_kaitou(f"  - {base_folder_name} 内の zip2 フォルダを削除しました。")
            except OSError as e:
                self.update_label_kaitou(f"  エラー: {base_folder_name} 内の zip2 フォルダの削除に失敗しました - {e}")
        elif has_outer_zip and not os.path.exists(zip2_path):
            self.update_label_kaitou(f"  - {base_folder_name} 内で処理されたzipファイルの中に、さらにzipファイルは存在しませんでした。zip2フォルダは作成されませんでした。")
        elif not has_outer_zip:
            self.update_label_kaitou(f"  - {base_folder_name} 内に処理対象のzipファイルが存在しませんでした。")
        self.update_label_kaitou(f"  - {base_folder_name} の処理が完了しました。\n")

    # 2. GeoJSON作成用のバッチファイルを作成
    def createPublicCoordinateBatch(self):
        if self.ikkatsu.isChecked():
            folder_name = self.folder_label.text().replace("選択中のフォルダ : ","")
        else:
            folder_name = QFileDialog.getExistingDirectory(self, "フォルダを選択")
        if folder_name:
            self.output_list.clear()
            for root, dirs, files in os.walk(folder_name):
                if os.path.basename(root).lower() == 'xml':
                    public_xml_files_for_batch = []
                    for file in files:
                        if file.lower().endswith('.xml') and '任意座標系' not in os.path.dirname(os.path.join(root, file)):
                            public_xml_files_for_batch.append(os.path.join(root, file))
                    if public_xml_files_for_batch:
                        mojxml2geojson = self.lineEdit_mojxml2geojson.text()
                        batch_content = "cd C:\ncd " + mojxml2geojson.replace("\r", "") + "\n"
                        for xml_file in public_xml_files_for_batch:
                            batch_content += f"mojxml2geojson -e -u -d -x {xml_file}\n"
                        batch_file_path = os.path.join(root, "convert_geojson.bat")
                        try:
                            with open(batch_file_path, "w", encoding="utf-8") as f:
                                f.write(batch_content)
                            if not self.ikkatsu.isChecked():
                                self.output_list.addItem(f"'{os.path.dirname(os.path.join(root, file))}' に '{os.path.basename(batch_file_path)}' を作成しました。")
                                self.output_list.scrollToBottom()
                        except Exception as e:
                            QMessageBox.critical(self, 'エラー', f"フォルダ '{os.path.basename(root)}' へのバッチファイル作成に失敗しました: {e}")
                    else:
                        QMessageBox.information(self, 'バッチファイル作成', f"フォルダ '{os.path.basename(root)}' に公共座標のXMLファイルが見つかりませんでした。")
            self.format1.setChecked(True)
            if self.ikkatsu2.isChecked():
                self.run_selected_batches()

    # 2-2. GeoJSON作成用のバッチファイルを作成
    def createPublicCoordinateBatch2(self):
        folder_name = QFileDialog.getExistingDirectory(self, "フォルダを選択")
        self.folder_label.setText(f'選択中のフォルダ : {folder_name}')
        if folder_name:
            self.output_list.clear()
            for root, dirs, files in os.walk(folder_name):
                if os.path.basename(root).lower() == 'xml':
                    public_xml_files_for_batch = []
                    for file in files:
                        if file.lower().endswith('.xml') and '任意座標系' not in os.path.dirname(os.path.join(root, file)):
                            public_xml_files_for_batch.append(os.path.join(root, file))
                    if public_xml_files_for_batch:
                        mojxml2geojson = self.lineEdit_mojxml2geojson.text()
                        batch_content = "cd C:\ncd " + mojxml2geojson.replace("\r", "") + "\n"
                        for xml_file in public_xml_files_for_batch:
                            batch_content += f"mojxml2geojson -e -u -d -x {xml_file}\n"
                        batch_file_path = os.path.join(root, "convert_geojson.bat")
                        try:
                            with open(batch_file_path, "w", encoding="utf-8") as f:
                                f.write(batch_content)
                            if not self.ikkatsu2.isChecked():
                                self.output_list.addItem(f"'{os.path.dirname(os.path.join(root, file))}' に '{os.path.basename(batch_file_path)}' を作成しました。")
                                self.output_list.scrollToBottom()
                        except Exception as e:
                            QMessageBox.critical(self, 'エラー', f"フォルダ '{os.path.basename(root)}' へのバッチファイル作成に失敗しました: {e}")
                    else:
                        QMessageBox.information(self, 'バッチファイル作成', f"フォルダ '{os.path.basename(root)}' に公共座標のXMLファイルが見つかりませんでした。")
            self.format1.setChecked(True)
            if self.ikkatsu2.isChecked():
                self.run_selected_batches2()

    # 3. FlatGeobuf用バッチファイル作成
    def generate_batch_files(self):
        if self.ikkatsu.isChecked():
            folder_path = self.folder_label.text().replace("選択中のフォルダ : ","")
        else:
            folder_path = QFileDialog.getExistingDirectory(self, "フォルダを選択")
        if folder_path:
            self.root_dir = folder_path
            self.folder_label.setText(f'選択中のフォルダ : {folder_path}')
            self.output_list.clear()
            found_xml_folders = False
            for root, dirs, files in os.walk(self.root_dir):
                if os.path.basename(root) == 'xml':
                    found_xml_folders = True
                    self.output_list.addItem(f"処理中の XML フォルダ: {root}")
                    self.output_list.scrollToBottom()
                    python = self.lineEdit_python.text().replace("\r", "")
                    ogrmerge = self.lineEdit_ogrmerge.text().replace("\r", "")
                    # 親フォルダ名を取得
                    parent_folder_name = os.path.basename(os.path.dirname(root))
                    # kukaku フォルダ用のバッチファイル作成
                    kukaku_folder_path = os.path.join(root, 'kukaku')
                    kukaku_batch_name = f"{parent_folder_name}_kukaku_fgb.bat"
                    kukaku_batch_path = os.path.join(root, kukaku_batch_name)
                    kukaku_input_path = os.path.join(kukaku_folder_path, '*.geojson')
                    kukaku_output_fgb_name = f"{parent_folder_name}_kukaku.fgb"
                    kukaku_batch_content = f"""{python} {ogrmerge} -f FlatGeobuf -single -skipfailures -t_srs EPSG:4326 -o "{os.path.join(root, kukaku_output_fgb_name)}" "{kukaku_input_path}" """
                    try:
                        os.makedirs(os.path.dirname(kukaku_batch_path), exist_ok=True)
                        with open(kukaku_batch_path, 'w', encoding='utf-8') as f:
                            f.write(kukaku_batch_content)
                        self.output_list.addItem(f"バッチファイルを生成しました: {kukaku_batch_path} (出力: {kukaku_output_fgb_name})")
                        self.output_list.scrollToBottom()
                    except Exception as e:
                        error_message = f"kukaku 用バッチファイルの作成に失敗しました: {e}"
                        QMessageBox.critical(self, "エラー", error_message)
                        traceback.print_exc()
                    # chiban フォルダ用のバッチファイル作成
                    chiban_folder_path = os.path.join(root, 'chiban')
                    chiban_batch_name = f"{parent_folder_name}_chiban_fgb.bat"
                    chiban_batch_path = os.path.join(root, chiban_batch_name)
                    chiban_input_path = os.path.join(chiban_folder_path, '*daihyo.geojson')
                    chiban_output_fgb_name = f"{parent_folder_name}_chiban.fgb"
                    chiban_batch_content = f"""{python} {ogrmerge} -f FlatGeobuf -single -skipfailures -t_srs EPSG:4326 -o "{os.path.join(root, chiban_output_fgb_name)}" "{chiban_input_path}" """
                    try:
                        os.makedirs(os.path.dirname(chiban_batch_path), exist_ok=True)
                        with open(chiban_batch_path, 'w', encoding='utf-8') as f:
                            f.write(chiban_batch_content)
                        self.output_list.addItem(f"バッチファイルを生成しました: {chiban_batch_path} (出力: {chiban_output_fgb_name})")
                        self.output_list.scrollToBottom()
                    except Exception as e:
                        error_message = f"chiban 用バッチファイルの作成に失敗しました: {e}"
                        QMessageBox.critical(self, "エラー", error_message)
                        traceback.print_exc()
            if not found_xml_folders:
                self.output_list.addItem('「xml」フォルダが見つかりませんでした。')
                self.output_list.scrollToBottom()
            else:
                self.output_list.addItem(f'\nバッチファイルの生成処理が完了しました。')
                self.output_list.scrollToBottom()
            self.format2.setChecked(True)
            if self.ikkatsu3.isChecked():
                self.run_selected_batches2()

    # 3-2. FlatGeobuf用バッチファイル作成
    def generate_batch_files2(self):
        folder_path = QFileDialog.getExistingDirectory(self, "フォルダを選択")
        if folder_path:
            self.root_dir = folder_path
            self.folder_label.setText(f'選択中のフォルダ : {folder_path}')
            self.output_list.clear()
            found_xml_folders = False
            for root, dirs, files in os.walk(self.root_dir):
                if os.path.basename(root) == 'xml':
                    found_xml_folders = True
                    self.output_list.addItem(f"処理中の XML フォルダ: {root}")
                    self.output_list.scrollToBottom()
                    python = self.lineEdit_python.text().replace("\r", "")
                    ogrmerge = self.lineEdit_ogrmerge.text().replace("\r", "")
                    # 親フォルダ名を取得
                    parent_folder_name = os.path.basename(os.path.dirname(root))[3:] #os.path.basename(os.path.dirname(root))
                    # kukaku フォルダ用のバッチファイル作成
                    kukaku_folder_path = os.path.join(root, 'kukaku')
                    kukaku_batch_name = f"{parent_folder_name}_kukaku_fgb.bat"
                    kukaku_batch_path = os.path.join(root, kukaku_batch_name)
                    kukaku_input_path = os.path.join(kukaku_folder_path, '*.geojson')
                    kukaku_output_fgb_name = f"{parent_folder_name}.fgb"
                    kukaku_batch_content = f"""{python} {ogrmerge} -f FlatGeobuf -single -skipfailures -t_srs EPSG:4326 -o "{os.path.join(root, kukaku_output_fgb_name)}" "{kukaku_input_path}" """
                    try:
                        os.makedirs(os.path.dirname(kukaku_batch_path), exist_ok=True)
                        with open(kukaku_batch_path, 'w', encoding='utf-8') as f:
                            f.write(kukaku_batch_content)
                        self.output_list.addItem(f"バッチファイルを生成しました: {kukaku_batch_path} (出力: {kukaku_output_fgb_name})")
                        self.output_list.scrollToBottom()
                    except Exception as e:
                        error_message = f"kukaku 用バッチファイルの作成に失敗しました: {e}"
                        QMessageBox.critical(self, "エラー", error_message)
                        traceback.print_exc()
                    # chiban フォルダ用のバッチファイル作成
                    chiban_folder_path = os.path.join(root, 'chiban')
                    chiban_batch_name = f"{parent_folder_name}_chiban_fgb.bat"
                    chiban_batch_path = os.path.join(root, chiban_batch_name)
                    chiban_input_path = os.path.join(chiban_folder_path, '*daihyo.geojson')
                    chiban_output_fgb_name = f"{parent_folder_name}_chiban.fgb"
                    chiban_batch_content = f"""{python} {ogrmerge} -f FlatGeobuf -single -skipfailures -t_srs EPSG:4326 -o "{os.path.join(root, chiban_output_fgb_name)}" "{chiban_input_path}" """
                    try:
                        os.makedirs(os.path.dirname(chiban_batch_path), exist_ok=True)
                        with open(chiban_batch_path, 'w', encoding='utf-8') as f:
                            f.write(chiban_batch_content)
                        self.output_list.addItem(f"バッチファイルを生成しました: {chiban_batch_path} (出力: {chiban_output_fgb_name})")
                        self.output_list.scrollToBottom()
                    except Exception as e:
                        error_message = f"chiban 用バッチファイルの作成に失敗しました: {e}"
                        QMessageBox.critical(self, "エラー", error_message)
                        traceback.print_exc()
            if not found_xml_folders:
                self.output_list.addItem('「xml」フォルダが見つかりませんでした。')
                self.output_list.scrollToBottom()
            else:
                self.output_list.addItem(f'\nバッチファイルの生成処理が完了しました。')
                self.output_list.scrollToBottom()
            self.format2.setChecked(True)
            if self.ikkatsu3.isChecked():
                self.run_selected_batches2()

    # 4. バッチファイル並列処理 GeoJSON用
    def run_selected_batches(self):
        if self.ikkatsu.isChecked():
            if not self.ikkatsu2.isChecked():
                return
            self.format1.setChecked(True)
            folder = self.folder_label.text().replace("選択中のフォルダ : ","")
        elif self.ikkatsu3.isChecked():
            self.format2.setChecked(True)
            folder = self.folder_label.text().replace("選択中のフォルダ : ","")
        else:
            # xmlフォルダ内のgeojsonバッチファイルを検索する
            folder = QFileDialog.getExistingDirectory(self, "フォルダを選択")
        if folder:
            self.folder_label.setText(f"選択中のフォルダ : {folder}")
            self.batch_list.clear()
            self.batch_files = []
            if self.format1.isChecked():
                fmt = "geojson"
            else:
                fmt = "fgb"
            for item in os.listdir(folder):
                item_path = os.path.join(folder, item)
                if os.path.isdir(item_path):
                    xml_folder = os.path.join(item_path, "xml")
                    if os.path.isdir(xml_folder):

                        kukaku_folder = os.path.join(xml_folder, 'kukaku')
                        chiban_folder = os.path.join(xml_folder, 'chiban')
                        os.makedirs(kukaku_folder, exist_ok=True)
                        os.makedirs(chiban_folder, exist_ok=True)

                        for batch_file in os.listdir(xml_folder):
                            if fmt in batch_file.lower() and batch_file.lower().endswith(".bat"):
                                batch_file_path = os.path.join(xml_folder, batch_file)
                                self.batch_files.append(batch_file_path)
                                self.batch_list.addItem(batch_file_path)
            if self.batch_files:
                pass
            else:
                self.output_list.addItem(fmt + " を含むバッチファイルは見つかりませんでした。")
                self.output_list.scrollToBottom()
        # 選択されたバッチファイルを同時に実行する
        self.output_list.clear()
        self.threads = []
        for filepath in self.batch_files:
            thread = BatchRunner(filepath)
            thread.finished.connect(self.update_output)
            self.threads.append(thread)
            thread.start()

    # 4. バッチファイル並列処理～ログ～ GeoJSON用
    def update_output(self, filepath, exit_code):
        # バッチファイルの実行結果をログに出力する
        if exit_code == 0:
            self.output_list.addItem(f"{filepath} は正常に終了しました。")
            self.output_list.scrollToBottom()
            if len(self.batch_list) == len(self.output_list) and self.ikkatsu.isChecked():
                self.batch_list.clear()
                self.output_list.clear()
                self.generate_batch_files()
        else:
            self.output_list.addItem(f"エラー: {filepath} は終了コード {exit_code} で終了しました。")
            self.output_list.scrollToBottom()

    # 4-2. バッチファイル並列処理 FlatGeobuf用
    def run_selected_batches2(self):
        folder = self.folder_label.text().replace("選択中のフォルダ : ","")
        if folder:
            self.folder_label.setText(f"選択中のフォルダ : {folder}")
            self.batch_list.clear()
            self.batch_files = []
            if self.format1.isChecked():
                fmt = "geojson"
            else:
                fmt = "fgb"
            for item in os.listdir(folder):
                item_path = os.path.join(folder, item)
                if os.path.isdir(item_path):
                    xml_folder = os.path.join(item_path, "xml")
                    if os.path.isdir(xml_folder):
                        for batch_file in os.listdir(xml_folder):
                            if fmt in batch_file.lower() and batch_file.lower().endswith(".bat"):
                                batch_file_path = os.path.join(xml_folder, batch_file)
                                self.batch_files.append(batch_file_path)
                                self.batch_list.addItem(batch_file_path)
            if self.batch_files:
                pass
            else:
                self.output_list.addItem(fmt + " を含むバッチファイルは見つかりませんでした。")
                self.output_list.scrollToBottom()
        # 選択されたバッチファイルを同時に実行する
        self.output_list.clear()
        self.threads = []
        for filepath in self.batch_files:
            thread = BatchRunner(filepath)
            thread.finished.connect(self.update_output2)
            self.threads.append(thread)
            thread.start()

    # 4-2. バッチファイル並列処理～ログ～ FlatGeobuf用
    def update_output2(self, filepath, exit_code):
        # バッチファイルの実行結果をログに出力する
        if exit_code == 0:
            self.output_list.addItem(f"{filepath} は正常に終了しました。")
            self.output_list.scrollToBottom()
            if self.ikkatsu4.isChecked():
                self.convert_pmtiles()
        else:
            self.output_list.addItem(f"エラー: {filepath} は終了コード {exit_code} で終了しました。")
            self.output_list.scrollToBottom()

    # 5. FlatGeobufファイル移動 ＆ PMTiles作成
    def convert_pmtiles(self):
        self.savesetting()
        if self.ikkatsu4.isChecked():
            folder = self.folder_label.text().replace("選択中のフォルダ : ","")
        elif self.ikkatsu3.isChecked():
            folder = QFileDialog.getExistingDirectory(self, "フォルダを選択")
            folder = self.folder_label.text().replace("選択中のフォルダ : ","")
        if folder:
            self.output_list.clear()
            kukaku_folder = os.path.join(folder, 'kukaku')
            chiban_folder = os.path.join(folder, 'chiban')
            os.makedirs(kukaku_folder, exist_ok=True)
            os.makedirs(chiban_folder, exist_ok=True)
            self.output_list.addItem('FlatGeobufファイルを移動します...')
            self.output_list.scrollToBottom()
            for item in os.listdir(folder):
                subdirectory_path = os.path.join(folder, item)
                if os.path.isdir(subdirectory_path):
                    xml_folder_path = os.path.join(subdirectory_path, 'xml')
                    if os.path.isdir(xml_folder_path):
                        for filename in os.listdir(xml_folder_path):
                            if filename.endswith('.fgb'):
                                source_path = os.path.join(xml_folder_path, filename)
                                if 'chiban' in filename:
                                    destination_path = os.path.join(chiban_folder, filename)
                                    try:
                                        shutil.move(source_path, destination_path)
                                        self.output_list.addItem(f'移動: {filename} -> {chiban_folder}')
                                        self.output_list.scrollToBottom()
                                    except Exception as e:
                                        self.output_list.addItem(f'エラー: {filename} の移動に失敗 - {e}')
                                        self.output_list.scrollToBottom()
                                else:
                                    destination_path = os.path.join(kukaku_folder, filename)
                                    try:
                                        shutil.move(source_path, destination_path)
                                        self.output_list.addItem(f'移動: {filename} -> {kukaku_folder}')
                                        self.output_list.scrollToBottom()
                                    except Exception as e:
                                        self.output_list.addItem(f'エラー: {filename} の移動に失敗 - {e}')
                                        self.output_list.scrollToBottom()
            self.output_list.addItem('フFlatGeobufファイルの移動が完了しました。')
            self.output_list.scrollToBottom()
            # PMTiles作成
            root_folder = folder.replace(":", "").lower()
            ubuntu = self.ubuntu.text()
            # 区画（ファイル名は、zenkoku.pmtiles）
            option = f'tippecanoe -l moj_map -rg50000 -z16 -Z14 --no-tile-size-limit --no-line-simplification -o /mnt/{root_folder}/kukaku/zenkoku.pmtiles /mnt/{root_folder}/kukaku/*.fgb'
            exec_command = f"wsl.exe -d {ubuntu} {option}"
            try:
                # コマンドを新しいウィンドウで起動
                process = subprocess.Popen(
                    ["cmd", "/c", exec_command],
                    creationflags=subprocess.CREATE_NEW_CONSOLE
                )
                self.output_list.addItem(f"")
                self.output_list.addItem(f"{option} を実行します。。")
                self.output_list.scrollToBottom()
                # 必要であれば、プロセスの終了を待機
                # process.wait()
                # print("tippecanoeの処理が完了しました。")
            except FileNotFoundError:
                print("wsl.exeが見つかりませんでした。WSLが正しくインストールされているか確認してください。")
            except Exception as e:
                print(f"予期しないエラーが発生しました: {e}")
            # 地番（ファイル名は、chiban15.pmtiles）
            option = f'tippecanoe -l chiban -rg50000 -z15 -Z15 --no-tile-size-limit -o /mnt/{root_folder}/chiban/chiban15.pmtiles /mnt/{root_folder}/chiban/*.fgb'
            exec_command = f"wsl.exe -d {ubuntu} {option}"
            try:
                # コマンドを新しいウィンドウで起動
                process = subprocess.Popen(
                    ["cmd", "/c", exec_command],
                    creationflags=subprocess.CREATE_NEW_CONSOLE
                )
                self.output_list.addItem(f"{option} を実行します。。")
                self.output_list.scrollToBottom()
                # 必要であれば、プロセスの終了を待機
                # process.wait()
                # print("tippecanoeの処理が完了しました。")
            except FileNotFoundError:
                print("wsl.exeが見つかりませんでした。WSLが正しくインストールされているか確認してください。")
            except Exception as e:
                print(f"予期しないエラーが発生しました: {e}")
            # 地番（ファイル名は、chiban16.pmtiles）
            option = f'tippecanoe -l chiban -rg50000 -z16 -Z16 --no-tile-size-limit -o /mnt/{root_folder}/chiban/chiban16.pmtiles /mnt/{root_folder}/chiban/*.fgb'
            exec_command = f"wsl.exe -d {ubuntu} {option}"
            try:
                # コマンドを新しいウィンドウで起動
                process = subprocess.Popen(
                    ["cmd", "/c", exec_command],
                    creationflags=subprocess.CREATE_NEW_CONSOLE
                )
                self.output_list.addItem(f"{option} を実行します。。")
                self.output_list.scrollToBottom()
                # 必要であれば、プロセスの終了を待機
                # process.wait()
                # print("tippecanoeの処理が完了しました。")
            except FileNotFoundError:
                print("wsl.exeが見つかりませんでした。WSLが正しくインストールされているか確認してください。")
            except Exception as e:
                print(f"予期しないエラーが発生しました: {e}")
            # 地番（ファイル名は、chiban17.pmtiles）
            option = f'tippecanoe -l chiban -rg50000 -z17 -Z17 --no-tile-size-limit -o /mnt/{root_folder}/chiban/chiban17.pmtiles /mnt/{root_folder}/chiban/*.fgb'
            exec_command = f"wsl.exe -d {ubuntu} {option}"
            try:
                # コマンドを新しいウィンドウで起動
                process = subprocess.Popen(
                    ["cmd", "/c", exec_command],
                    creationflags=subprocess.CREATE_NEW_CONSOLE
                )
                self.output_list.addItem(f"{option} を実行します。。")
                self.output_list.scrollToBottom()
                # 必要であれば、プロセスの終了を待機
                # process.wait()
                # print("tippecanoeの処理が完了しました。")
            except FileNotFoundError:
                print("wsl.exeが見つかりませんでした。WSLが正しくインストールされているか確認してください。")
            except Exception as e:
                print(f"予期しないエラーが発生しました: {e}")
            # 地番（ファイル名は、chiban18.pmtiles）
            option = f'tippecanoe -l chiban -rg50000 -z18 -Z18 --no-tile-size-limit -o /mnt/{root_folder}/chiban/chiban18.pmtiles /mnt/{root_folder}/chiban/*.fgb'
            exec_command = f"wsl.exe -d {ubuntu} {option}"
            try:
                # コマンドを新しいウィンドウで起動
                process = subprocess.Popen(
                    ["cmd", "/c", exec_command],
                    creationflags=subprocess.CREATE_NEW_CONSOLE
                )
                self.output_list.addItem(f"{option} を実行します。")
                self.output_list.scrollToBottom()
                # 必要であれば、プロセスの終了を待機
                process.wait()
                QMessageBox.information(self, 'PMTiles作成',"tippecanoeの処理が完了しました。")
            except FileNotFoundError:
                print("wsl.exeが見つかりませんでした。WSLが正しくインストールされているか確認してください。")
            except Exception as e:
                print(f"予期しないエラーが発生しました: {e}")

    # 公共座標XMLと任意座標XMLの仕分け
    def processXmlFiles(self):
        # ルートフォルダを指定する場合
        if self.ikkatsu.isChecked():
            folder = self.folder_label.text().replace("選択中のフォルダ : ","")
        else:
            folder = QFileDialog.getExistingDirectory(self, 'フォルダを選択')
            self.folder_label.setText(f'選択中のフォルダ : {folder}')
        if folder:
            self.selected_folder = folder
            search_string = '任意座標系'
            moved_total = 0
            all_processed_files = {}
            self.public_coordinate_xml_files = [] # 公共座標XMLファイルのリスト
            for root, dirs, files in os.walk(self.selected_folder):
                if os.path.basename(root).lower() == 'xml':
                    xml_files = [f for f in files if f.lower().endswith('.xml')]
                    total_files_in_folder = len(xml_files)
                    self.progress_bar.setMaximum(self.progress_bar.maximum() + total_files_in_folder) # 全体のファイル数を更新
                    moved_in_current_folder = []
                    new_folder_path = os.path.join(root, search_string)
                    os.makedirs(new_folder_path, exist_ok=True)
                    for i, filename in enumerate(xml_files):
                        filepath = os.path.join(root, filename)
                        found_arbitrary = False
                        try:
                            with open(filepath, 'r', encoding='utf-8') as f:
                                for line_num, line in enumerate(f):
                                    if line_num < 10 and search_string in line:
                                        found_arbitrary = True
                                        break
                        except Exception as e:
                            QMessageBox.warning(self, '警告', f'{filename} の読み込み中にエラーが発生しました: {e}')
                            continue
                        if found_arbitrary:
                            try:
                                shutil.move(filepath, os.path.join(new_folder_path, filename))
                                moved_in_current_folder.append(filename)
                                moved_total += 1
                            except Exception as e:
                                QMessageBox.warning(self, '警告', f'{filename} の移動中にエラーが発生しました: {e}')
                        else:
                            self.public_coordinate_xml_files.append(filepath) # 公共座標のファイルをリストに追加
                        self.progress_bar.setValue(self.progress_bar.value() + 1)
                        QApplication.processEvents()
                    if moved_in_current_folder:
                        all_processed_files[root] = moved_in_current_folder
            if not self.ikkatsu.isChecked():
                message = f'{moved_total} 個のXMLファイルを移動しました。\n'
                for folder, files in all_processed_files.items():
                    message += f'"{folder}" フォルダ内の {len(files)} 個のファイルを "{os.path.join(folder, search_string)}" フォルダに移動しました。\n'
                    # 必要であれば、ファイル名も追記
                    # message += f'  - {", ".join(files)}\n'
                QMessageBox.information(self, '処理完了', message)
            self.progress_bar.setValue(0) # プログレスバーをリセット

    # ファイル集計
    def countFiles(self):
        folder_path = QFileDialog.getExistingDirectory(self, "フォルダを選択")
        self.folder_label.setText(f'選択中のフォルダ : {folder_path}')
        self.output_list.clear()
        self.output_list.addItem("------------------------------")
        self.output_list.addItem("ファイル数集計")
        self.output_list.addItem("------------------------------")
        if folder_path:
            counts = self.count_file_types(folder_path)
            result_text = "選択中のフォルダとそのサブフォルダ内のファイル数:\n"
            for folder, count in counts.items():
                self.output_list.addItem(f"フォルダ: {folder}")
                self.output_list.addItem(f"  .xml ファイル数: {count['xml']}")
                self.output_list.addItem(f"  .zip ファイル数: {count['zip']}")
                self.output_list.addItem(f"  .geojson ファイル数: {count['geojson']}")
                self.output_list.addItem(f"  .fgb ファイル数: {count['fgb']}")
                self.output_list.addItem(f"  .pmtiles ファイル数: {count['pmtiles']}")
                self.output_list.addItem(f"")
            # 結果をファイルに保存
            output_filepath = os.path.join(folder_path, "count.txt")
            try:
                with open(output_filepath, "w", encoding="utf-8") as f:
                    for folder, count in counts.items():
                        f.write(f"フォルダ: {folder}\n")
                        f.write(f"  .xml ファイル数: {count['xml']}\n")
                        f.write(f"  .zip ファイル数: {count['zip']}\n")
                        f.write(f"  .geojson ファイル数: {count['geojson']}\n")
                        f.write(f"  .fgb ファイル数: {count['fgb']}\n")
                        f.write(f"  .pmtile ファイル数: {count['pmtiles']}\n\n")
                QMessageBox.information(self, "ファイル数確認", f"集計結果を {output_filepath} に保存しました。")
            except Exception as e:
                QMessageBox.critical(self, "エラー", f"結果の保存に失敗しました: {e}")

    # ファイル集計
    def count_file_types(self, root_dir):
        file_counts = {}
        for foldername, subfolders, filenames in os.walk(root_dir):
            xml_count = 0
            csv_count = 0
            zip_count = 0
            geojson_count = 0
            fgb_count = 0
            pmtiles_count = 0
            for filename in filenames:
                if filename.lower().endswith(".xml"):
                    xml_count += 1
                elif filename.lower().endswith(".zip"):
                    zip_count += 1
                elif filename.lower().endswith(".geojson"):
                    geojson_count += 1
                elif filename.lower().endswith(".fgb"):
                    fgb_count += 1
                elif filename.lower().endswith(".pmtiles"):
                    pmtiles_count += 1
            file_counts[foldername] = {'xml': xml_count, 'zip': zip_count, 'geojson': geojson_count, 'fgb': fgb_count, 'pmtiles': pmtiles_count}
        return file_counts

    # 0.地図XMLのダウンロード～地図XMLリストダウンロード～
    def executeGetFileList(self):
        self.savesetting()
        self.file_list_widget.clear()
        # ファイルリストをG空間情報センターから取得し、リストボックスに表示する。
        apikey = self.apikey_edit.text()
        output_dir = os.path.dirname(__file__)
        year = self.year.text().replace("\r", "")
        if not output_dir:
            QMessageBox.warning(self, "警告", "出力先フォルダを指定してください。")
            return
        # PowerShellコマンドを構築（ファイルリストのみ取得）
        ps_command = ""
        if apikey:
            ps_command = f"""
                $headers = @{{'X-CKAN-API-Key' = '{apikey}'}}
                Invoke-WebRequest -Uri 'https://www.geospatial.jp/ckan/api/3/action/resource_search?query=name:-{year}.zip&limit=2022' -Headers $headers | ConvertFrom-Json | % {{ $_.result.results }} | Sort-Object name | % {{ $_.url }} | Out-File -FilePath '{output_dir}/xml_list.txt' -Encoding utf8
                Get-Content -Path '{output_dir}/xml_list.txt'
            """
        else:
            ps_command = f"""
                Invoke-WebRequest -Uri 'https://www.geospatial.jp/ckan/api/3/action/resource_search?query=name:-{year}.zip&limit=2022' | ConvertFrom-Json | % {{ $_.result.results }} | Sort-Object name | % {{ $_.url }} | Out-File -FilePath '{output_dir}/xml_list.txt' -Encoding utf8
                Get-Content -Path '{output_dir}/xml_list.txt'
            """
        # PowerShellプロセスを開始
        self.process_list = QProcess(self)
        self.process_list.setProcessChannelMode(QProcess.MergedChannels)  # 標準出力と標準エラー出力をマージ
        self.process_list.start("powershell.exe", ["-NoProfile", "-ExecutionPolicy", "Unrestricted", "-Command", ps_command])
        self.download_list_button.setEnabled(False)  # 実行中はボタンを無効化
        self.file_list_widget.clear() # リストボックスをクリア
        self.downloaded_list = [] # リストをクリア
        self.download_selected_button.setEnabled(False) # ファイルリスト取得中はダウンロードボタンを無効化
        # 終了シグナルにコールバック関数を接続
        self.process_list.finished.connect(self.onGetFileListFinished)
        self.process_list.errorOccurred.connect(self.onGetFileListErrorOccurred)
        self.process_list.readyReadStandardOutput.connect(self.readListOutput) # 標準出力を読み取るシグナルを接続

    def onGetFileListFinished(self):
        # ファイルリストの取得が終了したときに呼び出される。
        self.download_list_button.setEnabled(True) # 実行後にボタンを有効化
        exitCode = self.process_list.exitCode()
        exitStatus = self.process_list.exitStatus()
        if exitStatus == QProcess.NormalExit and exitCode == 0:
            QMessageBox.information(self, "成功", "利用可能なファイルリストを取得しました。")
            # ダウンロードしたリストをリストボックスに表示
            self.file_list_widget.addItems(self.downloaded_list)
            if self.file_list_widget.count() > 0:
                self.download_selected_button.setEnabled(True) # ファイルがあればダウンロードボタンを有効化
            self.downloaded_list = [] # リストをクリア
        else:
            error_message = self.process_list.readAllStandardError().data().decode("utf-8").strip()
            if not error_message:
                error_message = "不明なエラーが発生しました。"
            QMessageBox.critical(self, "エラー", f"ファイルリストの取得中にエラーが発生しました: {error_message} (終了コード: {exitCode})")
            self.downloaded_list = [] # エラー発生時もリストをクリア
            self.download_selected_button.setEnabled(False)

    def onGetFileListErrorOccurred(self, error):
        # ファイルリスト取得中にPowerShellエラーが発生した場合に呼び出される。
        error_message = "PowerShellエラー (ファイルリスト取得): " + str(error)
        QMessageBox.critical(self, "エラー", error_message)
        self.download_list_button.setEnabled(True)  # エラー発生後もボタンを有効にする
        self.downloaded_list = [] # エラー発生時もリストをクリア
        self.download_selected_button.setEnabled(False)

    def readListOutput(self):
        # ファイルリスト取得時のPowerShellの標準出力を読み取る。
        output = self.process_list.readAllStandardOutput().data().decode("utf-8").strip()
        if output:
            # 改行で分割してリストに追加
            self.downloaded_list.extend(output.splitlines())

    def updateDownloadButtonState(self):
        # リストボックスの選択状態に応じてダウンロードボタンの状態を更新する。
        if self.file_list_widget.selectedItems():
            self.download_selected_button.setEnabled(True)
        else:
            self.download_selected_button.setEnabled(False)

    # 地図XMLのダウンロード
    def startDownload(self):
        # 指定された出力先フォルダにダウンロードを開始する。
        selected_items = self.file_list_widget.selectedItems() # 選択されたアイテムを取得
        if not selected_items:
            QMessageBox.warning(self, "警告", "ダウンロードするファイルを選択してください。")
            return
        # 保存先フォルダを選択後にダウンロードを開始する。
        output_dir = QFileDialog.getExistingDirectory(self, "保存先フォルダを選択")
        if output_dir:
            self.folder_label.setText(f'選択中のフォルダ : {output_dir}')
        else:
            QMessageBox.warning(self, "警告", "保存先フォルダが選択されていません。")
        if not output_dir:
            QMessageBox.warning(self, "警告", "出力先フォルダを指定してください。")
            return
        self.output_list2.clear()
        self.output_dir = output_dir
        self.groupBox_3.setEnabled(False)
        self.download_status_text.setText("ダウンロードを開始...")
        self.total_downloads = len(selected_items)
        self.completed_downloads = 0
        self.download_queue = [item.text() for item in selected_items] # ダウンロードキューをURLのリストで作成
        self.active_downloads = 0
        self.workers = [] # ワーカーリストをクリア
        self.download_in_progress = True # ダウンロード処理中フラグを設定
        self.startNextDownload()

    def startNextDownload(self):
       # 次のダウンロードを開始する（同時ダウンロード数を制御）。
        while self.active_downloads < self.max_concurrent_downloads and self.download_queue:
            file_url = self.download_queue.pop(0)
            parsed_url = urllib.parse.urlparse(file_url)
            file_name = os.path.basename(parsed_url.path)
            output_path_temp = os.path.join(self.output_dir, file_name)
            worker = DownloadWorker(file_url, output_path_temp, file_name)
            worker.finished.connect(self.onFileDownloadFinished)
            self.workers.append(worker) # ワーカーをリストに追加
            worker.start()
            self.active_downloads += 1
            self.download_status_text.setText(f"{self.completed_downloads + self.active_downloads}/{self.total_downloads} ファイルをダウンロード中...")
        if not self.download_queue and self.active_downloads == 0 and self.download_in_progress:
            self.all_downloads_finished()
            self.download_in_progress = False # ダウンロード完了フラグをリセット

    def onFileDownloadFinished(self, file_url, temp_file_path, file_name, success, error):
        # 個々のファイルのダウンロードが終了したときに呼び出される。
        self.active_downloads -= 1
        self.completed_downloads += 1
        self.download_status_text.setText(f"{self.completed_downloads}/{self.total_downloads} ファイルをダウンロード完了")
        # 終了したワーカーをリストから削除 (念のため)
        for worker in list(self.workers):
            if not worker.isRunning():
                self.workers.remove(worker)
                worker.deleteLater() # Qtのリソース管理に従い、オブジェクトを破棄
        if success:
            self.moveDownloadedFile(temp_file_path, file_name) # ファイル移動処理を個別のメソッドに
        else:
            #self.output_list2.addItem(f"{file_name} のダウンロード中にエラーが発生しました: {error}")
            self.output_list2.addItem(f"{file_url}")
        self.startNextDownload()

    def moveDownloadedFile(self, temp_file_path, file_name):
        # ダウンロード完了したファイルを適切なフォルダに移動する。
        # 排他制御を行う。
        self.move_mutex.lock()
        try:
            prefecture_code = file_name[:2]
            destination_folder = None
            for folder in self.prefecture_folders:
                if folder.startswith(prefecture_code):
                    destination_folder = os.path.join(self.output_dir, folder)
                    os.makedirs(destination_folder, exist_ok=True)
                    break
            if destination_folder:
                try:
                    shutil.move(temp_file_path, os.path.join(destination_folder, file_name))
                except Exception as e:
                    #QMessageBox.warning(self, "移動エラー", f"{file_name} の移動中にエラーが発生しました: {e}")
                    self.download_status_text.setText(f"{file_name} の移動中にエラーが発生しました: {e}")
            else:
                #QMessageBox.warning(self, "移動エラー", f"{file_name} に対応するフォルダが見つかりませんでした。")
                self.download_status_text.setText(f"{file_name} に対応するフォルダが見つかりませんでした。")
        finally:
            self.move_mutex.unlock()

    # 地図XMLのダウンロード完了
    def all_downloads_finished(self):
        # すべてのダウンロードが完了した後に最終的な処理を行う。
        self.groupBox_3.setEnabled(True)
        self.download_status_text.setText("ダウンロード完了")
        # (1) 「file_list_widget」で選択されている数を取得
        selected_count = len(self.file_list_widget.selectedItems())
        # (2) 「output_list2」に登録されている数（エラー数）を取得
        error_count = self.output_list2.count()
        # (3) ダウンロード成功数を算出
        downloaded_count = selected_count - error_count
        if error_count == 0:
            if self.ikkatsu0.isChecked():
                self.download_status_text.setText(f"選択した{downloaded_count}個のファイルのダウンロードおよび移動が完了しました。")
                self.tabWidget.setCurrentIndex(0)
                self.kaitou()
            else:
                QMessageBox.information(self, "完了", f"選択した{downloaded_count}個のファイルのダウンロードおよび移動が完了しました。")
        else:
            self.ikkatsu0.setChecked(False) # 連続処理を中断する
            QMessageBox.warning(self, "ダウンロード結果", f"ダウンロードが完了しました。\n成功: {downloaded_count}個\nエラー: {error_count}個 (エラーのあったファイルは選択されています。)")
            # (4) 「file_list_widget」の選択を全て解除
            self.file_list_widget.clearSelection()
            error_urls = []
            for i in range(self.output_list2.count()):
                error_urls.append(self.output_list2.item(i).text())
            # 「file_list_widget」と「output_list2」を照合して、一致するアイテムを選択状態にする
            for i in range(self.file_list_widget.count()):
                item = self.file_list_widget.item(i)
                if item.text() in error_urls:
                    item.setSelected(True)
            # スクリプトのフォルダに「error_list.txt」を作成
            script_dir = os.path.dirname(os.path.abspath(__file__))
            error_file_path = os.path.join(script_dir, "error_list.txt")
            try:
                with open(error_file_path, 'w', encoding='utf-8') as f:
                    for url in error_urls:
                        f.write(url + '\n')
            except Exception as e:
                QMessageBox.critical(self, "エラー", f"エラーリストの保存に失敗しました: {e}")
        # (5) ダウンロード完了後、ルートフォルダの直下に残っているzipファイルを削除
        import glob
        root_dir = os.getcwd()  # 現在の作業ディレクトリ（ルートフォルダと仮定）
        zip_files = glob.glob(os.path.join(root_dir, "*.zip"))
        for zip_file in zip_files:
            try:
                os.remove(zip_file)
            except Exception as e:
                print(f"警告: zipファイル '{zip_file}' の削除に失敗しました: {e}")

    # 全選択
    def select_all_items(self):
        # チェックボックスを全てオンにする
        for checkbox in self.checkboxes:
            checkbox.setChecked(True)
        # リストウィジェットのアイテムを全て選択する
        self.file_list_widget.selectAll()

    # 全解除
    def deselect_all_items(self):
        # チェックボックスを全てオフにする
        for checkbox in self.checkboxes:
            checkbox.setChecked(False)
        # リストウィジェットの選択を全て解除する
        self.file_list_widget.clearSelection()

    # 都道府県チェックボックスの選択・解除
    def checkbox_state_changed(self, state):
        sender_checkbox = self.sender()
        checkbox_suffix = sender_checkbox.objectName()[-2:]
        for i in range(self.file_list_widget.count()):
            item = self.file_list_widget.item(i)
            item_text = item.text()
            file_name = item_text.split('/')[-1]
            if file_name.startswith(checkbox_suffix):
                if state == Qt.Checked:
                    item.setSelected(True)
                else:
                    item.setSelected(False)

    # 検索用データ読み込み
    def read_csv_data(self):
        try:
            with open("search_data.csv", 'r', encoding='shift-jis') as csvfile:
                reader = csv.DictReader(csvfile)
                self.search_data = {row['市区町村']: row['コード番号'] for row in reader}
            #QMessageBox.information(self, "情報", "リストを読み込みました。")
        except FileNotFoundError:
            QMessageBox.critical(self, "エラー", "search_data.csv が見つかりません。")
            self.search_data = {}
        except Exception as e:
            QMessageBox.critical(self, "エラー", f"CSVファイルの読み込み中にエラーが発生しました: {e}")
            self.search_data = {}

    # 検索
    def search_and_select(self):
        search_word = self.kensaku_text.text()
        if not self.search_data:
            QMessageBox.warning(self, "警告", "先にリストを読み込んでください。")
            return
        if search_word in self.search_data:
            code_number = self.search_data[search_word]
            found_first = False  # 最初に見つかった項目を追跡
            for i in range(self.file_list_widget.count()):
                item_text = self.file_list_widget.item(i).text().split('/')[-1].split('-')[0]
                if item_text == code_number:
                    item = self.file_list_widget.item(i)
                    item.setSelected(True)
                    if not found_first:
                        self.file_list_widget.scrollToItem(item, hint=QListWidget.PositionAtTop)
                        found_first = True
        else:
            QMessageBox.information(self, "情報", "該当する市区町村は見つかりませんでした。")

    # GeoJSON→SIMAコンバータ/GeoJSON Viewer共通 ～フォルダ選択～
    def select_geojson_folder(self):
        folder_path = QFileDialog.getExistingDirectory(self, "GeoJSONファイルがあるフォルダを選択")
        if folder_path:
            self.populate_geojson_list(folder_path)

    # GeoJSON→SIMAコンバータ ～GeoJSONファイル選択～
    def process_selected_geojson(self):
        selected_items = self.geojson_list.selectedItems()
        if not selected_items:
            QMessageBox.warning(self, "警告", "処理するGeoJSONファイルをリストから選択してください。")
            return
        self.output_list3.clear()
        geojson_file = selected_items[0].text()
        self.create_sima_file_from_geojson(geojson_file) # 引数を渡す

    # GeoJSON→SIMAコンバータ ～SIMA変換・保存～
    def create_sima_file_from_geojson(self, geojson_filepath):
        """
        GeoJSONファイルを読み込み、SIMA形式に合わせてx（緯度）、y（経度）でoutput.simを作成します。
        （「地番区域」＋「地番」が同一なものを1つの区画とし、「地番」のみを出力 - ハイフンあり）。
        """
        output_content = [
            "G00,03,ConvertTool for Python,",
            "Z00,座標ﾃﾞｰﾀ,",
            "A00,",
        ]
        kukaku_coordinates_no = OrderedDict() # (地番区域, 地番) のタプルをキーとして、座標の連番リストを格納
        unique_coordinates = OrderedDict() # 座標と連番の対応を保持 (緯度, 経度) の順でキーを格納
        coordinate_counter = 1
        try:
            with open(geojson_filepath, 'r', encoding='utf-8') as f:
                geojson_data = geojson.load(f)
            # 座標データの抽出と重複排除、連番付与
            for feature in geojson_data.get('features', []):
                地番区域 = feature.get('properties', {}).get('地番区域', 'unknown')
                地番 = feature.get('properties', {}).get('地番', 'unknown')
                if feature.get('geometry'):
                    geometry_type = feature['geometry']['type']
                    coordinates = feature['geometry']['coordinates']
                    polygon_coordinates_list = []
                    kukaku_coordinates_no_list = [] # この区画の座標連番を保持するリスト
                    if geometry_type == 'MultiPolygon':
                        for polygon in coordinates:
                            if polygon and len(polygon) > 0:
                                polygon_coordinates_list.extend(polygon)
                    elif geometry_type == 'Polygon':
                        if coordinates and len(coordinates) > 0:
                            polygon_coordinates_list.extend(coordinates)
                    for linear_ring in polygon_coordinates_list:
                        for coord in linear_ring:
                            lat, lon = coord[1], coord[0] # (緯度, 経度)の順
                            coordinate_pair = (lat, lon)
                            if coordinate_pair not in unique_coordinates:
                                unique_coordinates[coordinate_pair] = coordinate_counter
                                output_content.append(f"A01,{coordinate_counter},{coordinate_counter},{lat},{lon},,")
                                coordinate_counter += 1
                            no = unique_coordinates.get(coordinate_pair)
                            if no is not None:
                                kukaku_coordinates_no_list.append(no)
                    key = (地番区域, 地番)
                    if key not in kukaku_coordinates_no:
                        kukaku_coordinates_no[key] = []
                    kukaku_coordinates_no[key].extend(kukaku_coordinates_no_list)
            output_content.append("A99,")
            output_content.append("Z00,区画ﾃﾞｰﾀ,")
            # 区画データの書き出し
            kukaku_counter = 1
            for key, nos_list in kukaku_coordinates_no.items():
                chiban_kukaku, chiban = key
                output_content.append(f"D00,{kukaku_counter},{chiban},2,")
                for no in nos_list:
                    output_content.append(f"B01,{no},{no},")
                output_content.append("D99,")
                kukaku_counter += 1
            file_dialog = QFileDialog(self)
            output_filepath, _ = file_dialog.getSaveFileName(
                self,
                "SIMAファイルの保存",
                "output.sim",
                "SIM Files (*.sim)"
            )
            if output_filepath:
                try:
                    with open(output_filepath, 'w', newline='\r\n', encoding='shift_jis') as outfile:
                        outfile.write('\n'.join(output_content))
                    QMessageBox.information(self, "完了", "SIMAファイルを作成しました。")
                    self.output_list3.addItem(f"SIMAファイルを作成しました: {output_filepath}")
                except Exception as e:
                    self.output_list3.addItem(f"ファイルの保存中にエラーが発生しました: {e}")
        except FileNotFoundError:
            self.output_list3.addItem(f"エラー: ファイル '{geojson_filepath}' が見つかりません。")
        except json.JSONDecodeError:
            self.output_list3.addItem(f"エラー: GeoJSONファイルの読み込みに失敗しました。ファイルが有効なJSON形式であることを確認してください。")
        except Exception as e:
            self.output_list3.addItem(f"予期せぬエラーが発生しました: {e}")

    # GeoJSON→DXFコンバータ ～DXF/SFC変換・保存～
    def convert_selected_geojson(self):
        selected_item = self.geojson_list.currentItem()
        if not selected_item:
            QMessageBox.critical(self, "エラー", "変換するGeoJSONファイルを選択してください。")
            return
        selected_geojson_path = selected_item.text()
        if "daihyo" in os.path.basename(selected_geojson_path).lower():
            default_sfc_filename = os.path.splitext(os.path.basename(selected_geojson_path))[0] + ".sfc"
            output_sfc_path, _ = QFileDialog.getSaveFileName(self, "SFCファイルを保存", default_sfc_filename, "SFC Files (*.sfc)")
            if not output_sfc_path:
                return
            try:
                with open(selected_geojson_path, 'r', encoding='utf-8') as f:
                    geojson_data = json.load(f)
                features = geojson_data.get('features', [])
                sfc_lines = [
                    "HEADER;",
                    "FILE_DESCRIPTION(('SCADEC level2 feature_mode'),",
                    "         '2;1');",
                    "FILE_NAME('',",
                    "         '',",
                    "         (''),",
                    "         (''),",
                    "         'SCADEC_API_Ver3.01',",
                    "         '',",
                    "         '');",
                    "FILE_SCHEMA(('ASSOCIATIVE_DRAUGHTING'));",
                    "ENDSEC;",
                    "DATA;",
                    "/*SXF",
                    "#10 = pre_defined_colour_feature(\'black\')",
                    "SXF*/",
                    "/*SXF",
                    "#20 = user_defined_colour_feature('0','0','0')",
                    "SXF*/",
                    "/*SXF",
                    "#30 = pre_defined_font_feature(\'continuous\')",
                    "SXF*/",
                    "/*SXF",
                    "#40 = width_feature('0.130000')",
                    "SXF*/",
                    "/*SXF",
                    "#50 = text_font_feature(\'ＭＳ ゴシック\')",
                    "SXF*/",
                    "/*SXF",
                    "#60 = composite_curve_org_feature('1','1','1','0')",
                    "SXF*/"
                ]
                x_coords = []
                y_coords = []
                feature_number = 70
                for feature in features:
                    properties = feature.get('properties', {})
                    jibango = properties.get('地番')
                    geometry = feature.get('geometry')
                    if jibango is not None and geometry and geometry.get('type') == 'Point' and len(geometry.get('coordinates')) >= 2:
                        coordinates = geometry.get('coordinates')
                        x_coord = coordinates[0]
                        y_coord = coordinates[1]
                        sfc_line = f"/*SXF\n#{feature_number} = text_string_feature('2','17','1',\'{jibango}\',\'{x_coord * 1000:.6f}\',\'{y_coord * 1000:.6f}\','10000.000000','15000.000000','0.000000','0.00000000000000','0.00000000000000','2','1')\nSXF*/"
                        sfc_lines.append(sfc_line)
                        x_coords.append(x_coord)
                        y_coords.append(y_coord)
                        feature_number += 10
                num_features = len(x_coords)
                sfc_lines.extend([
                    f"/*SXF\n#{(num_features * 10) + 70} = sfig_org_feature(\'地番図\','2')\nSXF*/",
                    f"/*SXF\n#{(num_features * 10) + 80} = externally_defined_hatch_feature('1',\'Area_control\','1','0','()')\nSXF*/",
                    f"/*SXF\n#{(num_features * 10) + 90} = sfig_org_feature(\'$$ATRU$$57$$背景色$$色$$255_255_255\','3')\nSXF*/"
                ])
                if x_coords:
                    avg_x = sum(x_coords) / len(x_coords)
                    avg_y = sum(y_coords) / len(y_coords)
                    sfc_locate_x = abs(avg_x - (1456 * 5)) / 10
                    sfc_locate_y = abs(avg_y - (1030 * 5)) / 10
                    sfc_lines.append(f"/*SXF\n#{(num_features * 10) + 100} = sfig_locate_feature('0',\'地番図\',\'{sfc_locate_x:.8f}\',\'{sfc_locate_y:.8f}\','0.00000000000000','0.00010000000000','0.00010000000000')\nSXF*/")
                else:
                    sfc_lines.append(f"/*SXF\n#100 = sfig_locate_feature('0',\'地番図\','0.00000000','0.00000000','0.00000000000000','0.00010000000000','0.00010000000000')\nSXF*/")
                sfc_lines.extend([
                    f"/*SXF\n#{(num_features * 10) + 110} = sfig_locate_feature('0',\'$$ATRU$$57$$背景色$$色$$255_255_255\','0.000000','0.000000','0.00000000000000','1.00000000000000','1.00000000000000')\nSXF*/",
                    f"/*SXF3\n#{(num_features * 10) + 120} = drawing_attribute_feature(\'\',\'\',\'\',\'\',\'\',\'\',\'\',\'\',\'\',\'\',\'\',\'\')\nSXF3*/",
                    f"/*SXF\n#{(num_features * 10) + 130} = drawing_sheet_feature(\'地番図\','9','1','1456','1030')\nSXF*/",
                    f"/*SXF\n#{(num_features * 10) + 140} = layer_feature(\'汎用Layer\','1')\nSXF*/",
                    f"/*SXF\n#{(num_features * 10) + 150} = layer_feature(\'地番名\','1')\nSXF*/",
                    "ENDSEC;"
                ])
                with open(output_sfc_path, 'w', encoding='shift-jis') as f:
                    f.write('\n'.join(sfc_lines))
                QMessageBox.information(self, "完了", f"'{selected_geojson_path}' から '{output_sfc_path}' を作成しました。")
            except FileNotFoundError:
                QMessageBox.critical(self, "エラー", f"ファイル '{selected_geojson_path}' が見つかりません。")
            except json.JSONDecodeError:
                QMessageBox.critical(self, "エラー", f"'{selected_geojson_path}' は有効なGeoJSONファイルではありません。")
            except Exception as e:
                QMessageBox.critical(self, "エラー", f"SFCファイルの作成中にエラーが発生しました: {e}")
        else:
            default_dxf_filename = os.path.splitext(os.path.basename(selected_geojson_path))[0] + ".dxf"
            output_dxf_path, _ = QFileDialog.getSaveFileName(self, "DXFファイルを保存", default_dxf_filename, "DXF Files (*.dxf)")
            if not output_dxf_path:
                return
            try:
                scale_factor = float(self.scale_edit.text())
                if scale_factor <= 0:
                    QMessageBox.warning(self, "警告", "拡大率は正の数で入力してください。")
                    return
            except ValueError:
                QMessageBox.warning(self, "警告", "拡大率には数値を入力してください。")
                return
            try:
                geojson_driver = ogr.GetDriverByName("GeoJSON")
                if geojson_driver is None:
                    QMessageBox.critical(self, "エラー", "GeoJSONドライバが見つかりません。")
                    return
                geojson_datasource = geojson_driver.Open(selected_geojson_path, 0)
                if geojson_datasource is None:
                    QMessageBox.critical(self, "エラー", f"GeoJSONファイル '{selected_geojson_path}' を開けませんでした。")
                    return
                geojson_layer = geojson_datasource.GetLayer()
                dxf_driver = ogr.GetDriverByName("DXF")
                if dxf_driver is None:
                    QMessageBox.critical(self, "エラー", "DXFドライバが見つかりません。")
                    return
                dxf_datasource = dxf_driver.CreateDataSource(output_dxf_path)
                if dxf_datasource is None:
                    QMessageBox.critical(self, "エラー", f"DXFファイル '{output_dxf_path}' を作成できませんでした。")
                    return
                source_srs = geojson_layer.GetSpatialRef()
                if source_srs is None:
                    source_srs = osr.SpatialReference()
                    source_srs.SetWellKnownGeogCS("WGS84")
                dxf_layer = dxf_datasource.CreateLayer("polygon_outlines", geom_type=ogr.wkbLineString, options=['LAYER_COLOR=7'])
                feature = geojson_layer.GetNextFeature()
                while feature:
                    geometry = feature.GetGeometryRef()
                    if geometry:
                        scaled_geometry = ogr.CreateGeometryFromWkt(geometry.ExportToWkt())
                        if scaled_geometry.GetGeometryName() == 'MULTIPOLYGON':
                            for i in range(scaled_geometry.GetGeometryCount()):
                                polygon = scaled_geometry.GetGeometryRef(i)
                                for j in range(polygon.GetGeometryCount()):
                                    exterior_ring = polygon.GetGeometryRef(j)
                                    line_string = ogr.Geometry(ogr.wkbLineString)
                                    for k in range(exterior_ring.GetPointCount()):
                                        x, y, z = exterior_ring.GetPoint(k)
                                        line_string.AddPoint(x * scale_factor, y * scale_factor, z * scale_factor if z else 0)
                                    if exterior_ring.GetPointCount() > 0:
                                        x, y, z = exterior_ring.GetPoint(0)
                                        line_string.AddPoint(x * scale_factor, y * scale_factor, z * scale_factor if z else 0)
                                    dxf_feature = ogr.Feature(dxf_layer.GetLayerDefn())
                                    dxf_feature.SetGeometry(line_string)
                                    dxf_layer.CreateFeature(dxf_feature)
                                    dxf_feature = None
                        elif scaled_geometry.GetGeometryName() == 'POLYGON':
                            for i in range(scaled_geometry.GetGeometryCount()):
                                exterior_ring = scaled_geometry.GetGeometryRef(i)
                                line_string = ogr.Geometry(ogr.wkbLineString)
                                for j in range(exterior_ring.GetPointCount()):
                                    x, y, z = exterior_ring.GetPoint(j)
                                    line_string.AddPoint(x * scale_factor, y * scale_factor, z * scale_factor if z else 0)
                                if exterior_ring.GetPointCount() > 0:
                                    x, y, z = exterior_ring.GetPoint(0)
                                    line_string.AddPoint(x * scale_factor, y * scale_factor, z * scale_factor if z else 0)
                                dxf_feature = ogr.Feature(dxf_layer.GetLayerDefn())
                                dxf_feature.SetGeometry(line_string)
                                dxf_layer.CreateFeature(dxf_feature)
                                dxf_feature = None
                    feature.Destroy()
                    feature = geojson_layer.GetNextFeature()
                geojson_datasource = None
                dxf_datasource = None
                QMessageBox.information(self, "完了", f"'{selected_geojson_path}' を拡大率 {scale_factor} で '{output_dxf_path}' (線のみ) に保存しました。")
            except Exception as e:
                QMessageBox.critical(self, "エラー", f"変換中にエラーが発生しました: {e}")


    # GeoJSON→KMLコンバータ ～KML変換・保存～
    def create_kml_with_simplekml(self, geojson_path, epsg_code, output_kml_path):
        """
        GeoJSONファイルから simplekml を使用して KML ファイルを生成します。

        Args:
            geojson_path (str): 入力 GeoJSON ファイルのパス。
            epsg_code (int): 変換元の平面直角座標系の EPSG コード。
            output_kml_path (str): 出力 KML ファイルのパス。
        """
        kml = simplekml.Kml()
        transformer = Transformer.from_crs(f"EPSG:{epsg_code}", "EPSG:4326", always_xy=True)

        with open(geojson_path, 'r', encoding='utf-8') as f:
            geojson_data = json.load(f)

        for feature in geojson_data['features']:
            properties = feature['properties']
            geometry_type = feature['geometry']['type']
            coordinates = feature['geometry']['coordinates']

            if geometry_type == 'MultiPolygon':
                for polygon_coords in coordinates:
                    outer_boundary = []
                    for ring in polygon_coords:
                        converted_ring = [transformer.transform(x, y) for x, y in ring]
                        outer_boundary.append(converted_ring)

                    if outer_boundary:
                        pol = kml.newpolygon(name=properties.get('地番', 'No Name'))
                        pol.outerboundaryis = outer_boundary[0]
                        pol.description = self.create_description_html(properties)
                        # ポリゴンの塗りつぶしを透明にする (アルファ値を 00 に設定)
                        pol.style.polystyle.fill = 0
                        # ポリゴンの線の色を白に設定 (任意)
                        pol.style.linestyle.color = 'ffffffff'

            # 代表点の処理
            representative_x = properties.get('代表点経度／Y座標')
            representative_y = properties.get('代表点緯度／X座標')
            jibann = properties.get('地番')
            if representative_x is not None and representative_y is not None and jibann:
                lon, lat = transformer.transform(representative_x, representative_y)
                pnt = kml.newpoint(name=jibann, coords=[(lon, lat)])
                # ピン（マーカー）を非表示にする (スケールを 0 に設定)
                pnt.style.iconstyle.scale = 0

        kml.save(output_kml_path)
        print(f"KMLファイルが {output_kml_path} に保存されました。")
        # Google Earthを起動
        if sys.platform == "win32":
            os.startfile(output_kml_path)

    # GeoJSON→KMLコンバータ
    def create_description_html(self, properties):
        """ポップアップに表示する HTML 形式の文字列を生成します。"""
        description = f"""
        <!DOCTYPE html>
        <html>
        <head>
        <meta charset="UTF-8">
        </head>
        <body>
        <p><b>所在:</b> {properties.get('地番区域', '')}</p>
        <p><b>地番:</b> {properties.get('地番', '')}</p>
        <p><b>座標系:</b> {properties.get('座標系', '')}</p>
        <p><b>地図番号:</b> {properties.get('地図番号', '')}</p>
        <p><b>精度区分:</b> {properties.get('精度区分', '')}</p>
        <p><b>座標値種別:</b> {properties.get('座標値種別', '')}</p>
        <p><b>測地系判別:</b> {properties.get('測地系判別', '')}</p>
        <p><b>縮尺分母:</b> {properties.get('縮尺分母', '')}</p>
        <p><b>地図XMLファイル:</b> {properties.get('地図XMLファイル', '')}</p>
        </body>
        </html>
        """
        return description

    # GeoJSON→KMLコンバータ
    def convert_to_kml(self):
        selected_item = self.geojson_list.currentItem()
        if not selected_item:
            QMessageBox.warning(self, '警告', '変換するGeoJSONファイルを選択してください。')
            return

        input_geojson_path = selected_item.text()
        output_kml_path, _ = QFileDialog.getSaveFileName(self, 'KMLファイルの保存先',
                                                        os.path.splitext(input_geojson_path)[0] + '.kml',
                                                        'KML files (*.kml)')
        if output_kml_path:
            selected_kei = int(self.kei.currentText())
            epsg_code = 2442 + selected_kei

            try:
                # クラス内のメソッドを呼び出す際は self. をつける
                self.create_kml_with_simplekml(input_geojson_path, epsg_code, output_kml_path)
                QMessageBox.information(self, '完了', f'KMLファイルが {output_kml_path} に保存されました。')

            except Exception as e:
                QMessageBox.critical(self, 'エラー', f'変換中にエラーが発生しました: {e}')

    # GeoJSON Viewer
    def populate_geojson_list(self, folder_path):
        self.geojson_list.clear()
        self.geojson_files = []
        for root, _, files in os.walk(folder_path):
            for file in files:
                if file.endswith(".geojson"):
                    full_path = os.path.join(root, file)
                    self.geojson_files.append(full_path)
                    self.geojson_list.addItem(full_path)
        if self.geojson_files:
            self.geojson_list.setCurrentRow(0)
            self.load_selected_geojson_data()
            self.geojson_list.currentRowChanged.connect(self.load_selected_geojson_data)
        else:
            QMessageBox.information(self, "情報", "指定されたフォルダにGeoJSONファイルが見つかりませんでした。")
            self.selected_geojson_data = None

    # GeoJSON Viewer
    def load_selected_geojson_data(self):
        selected_item = self.geojson_list.currentItem()
        if selected_item:
            selected_index = self.geojson_list.currentRow()
            file_path = self.geojson_files[selected_index]
            try:
                with open(file_path, 'r', encoding='utf-8') as f:
                    self.selected_geojson_data = json.load(f)
            except FileNotFoundError:
                QMessageBox.critical(self, "エラー", "GeoJSONファイルが見つかりませんでした。")
                self.selected_geojson_data = None
            except json.JSONDecodeError:
                QMessageBox.critical(self, "エラー", "GeoJSONファイルの形式が不正です。")
                self.selected_geojson_data = None
            except Exception as e:
                QMessageBox.critical(self, "エラー", f"GeoJSONファイルの読み込みに失敗しました：{e}")
                self.selected_geojson_data = None
        else:
            self.selected_geojson_data = None

    # GeoJSON Viewer
    def draw_selected_geojson(self):
        if self.selected_geojson_data:
            self.plot_window = PlotWindow(self.selected_geojson_data)
            self.plot_window.show()
        else:
            QMessageBox.warning(self, "警告", "描画するGeoJSONデータが選択されていません。")

    # XML→SIMAコンバート（基準点のみ）～　地図XMLファイル読込
    def open_xml(self):
        options = QFileDialog.Options()
        file_path, _ = QFileDialog.getOpenFileName(self, "地図XMLファイルを選択", "", "XML Files (*.xml);;All Files (*)", options=options)
        if file_path:
            self.convert_xml_to_sima(file_path)
        else:
            QMessageBox.information(self, "キャンセル", "ファイル選択がキャンセルされました。\n処理を中止します。", QMessageBox.Ok)

    # XML→SIMAコンバート（基準点のみ）～　コンバート
    def convert_xml_to_sima(self, xml_file_path):
        try:
            tree = ET.parse(xml_file_path)
            root = tree.getroot()
            # XML名前空間の定義
            namespaces = {
                'tizu': 'http://www.moj.go.jp/MINJI/tizuxml',
                'zmn': 'http://www.moj.go.jp/MINJI/tizuzumen'
            }
            points_data = {}
            # 空間属性から座標を抽出
            for gm_point in root.findall('.//zmn:GM_Point', namespaces):
                point_id = gm_point.get('id')
                x = gm_point.find('.//zmn:X', namespaces)
                y = gm_point.find('.//zmn:Y', namespaces)
                if point_id and x is not None and y is not None:
                    points_data[point_id] = {
                        'X': float(x.text),
                        'Y': float(y.text)
                    }
            sima_lines = []
            sima_text =""
            sima_lines.append("G00,03,ConvertTool for Python,")
            sima_text = "G00,03,ConvertTool for Python,\n"
            sima_lines.append("Z00,座標ﾃﾞｰﾀ,")
            sima_text = sima_text + "Z00,座標ﾃﾞｰﾀ,\n"
            sima_lines.append("A00,")
            sima_text = sima_text + "A00,\n"
            point_number = 1
            # 主題属性から基準点情報を抽出し、SIMA形式の行を生成
            for kijunten in root.findall('.//tizu:基準点', namespaces):
                name_elem = kijunten.find('tizu:名称', namespaces)
                shape_elem = kijunten.find('tizu:形状', namespaces)
                type_elem = kijunten.find('tizu:基準点種別', namespaces)
                if name_elem is not None and shape_elem is not None and type_elem is not None:
                    point_id_ref = shape_elem.get('idref')
                    if point_id_ref and point_id_ref in points_data:
                        if self.check_shubetsu.isChecked():
                            point_name = f"{type_elem.text} {name_elem.text}"
                        else:
                            point_name = f"{name_elem.text}"
                        x_coord = points_data[point_id_ref]['X']
                        y_coord = points_data[point_id_ref]['Y']
                        sima_lines.append(f"A01,{point_number},{point_name},{x_coord},{y_coord},,")
                        sima_text = sima_text + f"A01,{point_number},{point_name},{x_coord},{y_coord},,\n"
                        point_number += 1
            sima_lines.append("A99,")
            sima_text = sima_text + "A99,\n"
            self.save_sima_file(xml_file_path, sima_lines)
            self.sima_text.setText(sima_text)
        except Exception as e:
            QMessageBox.critical(self, "エラー", f"ファイルの読み込みまたは変換中にエラーが発生しました。\nエラー内容: {e}", QMessageBox.Ok)

    def save_sima_file(self, original_xml_path, sima_lines):
        options = QFileDialog.Options()
        default_file_name = os.path.splitext(os.path.basename(original_xml_path))[0] + ".sim"
        save_path, _ = QFileDialog.getSaveFileName(self, "SIMAファイルを保存", default_file_name, "SIMA Files (*.sim);;All Files (*)", options=options)
        if save_path:
            try:
                with open(save_path, 'w', encoding='shift_jis') as f: # SIMAファイルはShift_JISエンコーディングが一般的
                    for line in sima_lines:
                        f.write(line + '\n')
                QMessageBox.information(self, "変換完了", f"SIMAファイルが正常に保存されました:\n{save_path}", QMessageBox.Ok)
            except Exception as e:
                QMessageBox.critical(self, "保存エラー", f"SIMAファイルの保存中にエラーが発生しました。\nエラー内容: {e}", QMessageBox.Ok)
        else:
            QMessageBox.information(self, "保存キャンセル", "SIMAファイルの保存がキャンセルされました。", QMessageBox.Ok)


if __name__ == '__main__':
    app = QApplication(sys.argv)
    win = Dialog()
    sys.exit(app.exec_())
    converter = GeoJSONConverter()
    #window = GeoJSONPlotView()
    #window.show()
    self.savesetting()