# coding: utf-8

from geojson import Feature, FeatureCollection, MultiPolygon
import geojson
import itertools

import package.ConvertDatum as ConvertDatum
import package.PointOnSurface as PointOnSurface

import os
import sys
import argparse

import pathlib
import re  # 正規表現モジュールを追加


def get_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('FILE_PATH')
    parser.add_argument('-e', '--exclude', help='地番が地図外、別図、区域外、調査外の筆を除外して出力', action='store_true')
    parser.add_argument('-u', '--untreated', help='任意座標系を除外して出力', action='store_true')
    parser.add_argument('-t', '--treated', help='任意座標系のみ出力', action='store_true')
    parser.add_argument('-d', '--daihyo', help='代表点を別ファイルで出力', action='store_true')
    parser.add_argument('-x', '--xy', help='座標変換しない', action='store_true')
    parser.add_argument('-c', '--chiban', help='代表点のみ出力', action='store_true')
    args = parser.parse_args()
    return args


def MojXMLtoGeoJSON(mojObj: dict, exclude_flag, untreated_flag, treated_flag, daihyo_flag, src_file, convert_flag, chiban_flag):
    args = get_args()
    srcFile = args.FILE_PATH

    IdHudeTable = mojObj['mojXmlPolygon']['IdHudeTable']
    IdSurfaceTable = mojObj['mojXmlPolygon']['IdSurfaceTable']

    feature_list = []
    feature2_list = []
    daihyo_geojson_str = None # daihyo付きgeojsonの文字列を格納する変数

    # 任意座標系のCRS
    #-----------------------------------------------
    crs_name = mojObj['named_crs']
    if crs_name is None:
        epsg = '2455'
    else:
        epsg = crs_name.replace('urn:ogc:def:crs:EPSG::','')
    #-----------------------------------------------

    # 代表点のみ出力の場合は、代表点を別ファイルで出力を有効にする
    #-----------------------------------------------
    if chiban_flag is True:
        daihyo_flag = True

    if daihyo_flag is True:
        if convert_flag is True:
            daihyo_text = '{"type":"FeatureCollection","name":"daihyo","crs": {"type": "name","properties": {"name": "urn:ogc:def:crs:EPSG::' + epsg + '"}},"features": ['
        else:
            daihyo_text = '{"type":"FeatureCollection","name":"daihyo","crs": {"type": "name","properties": {"name": "urn:ogc:def:crs:EPSG::4326"}},"features": ['
    #-----------------------------------------------

    for key in IdHudeTable.keys():
        hude = IdHudeTable[key]
        ref_gm_surface = hude['形状']['@idref']

        hude_property = setExistProperties(hude,
                                            ['地図番号', '縮尺分母', '大字名', '丁目名', '小字名', '予備名', '地番', '精度区分', '座標値種別', '筆界未定構成筆'],
                                            ['地図番号', '縮尺分母', '大字名', '丁目名', '小字名', '予備名', '地番', '精度区分', '座標値種別', '筆界未定構成筆'])

        # 地番が地区外、別図のデータを出力しない
        if exclude_flag is True:
            if ('地区外' in hude_property['地番']) or ('別図' in hude_property['地番']) or ('区域外' in hude_property['地番']) or ('調査外' in hude_property['地番']):
                continue

        # 【追加】筆界未定地の場合の処理
        if '筆界未定地' in hude_property['地番'] and '筆界未定構成筆' in hude_property and hude_property['筆界未定構成筆'] is not None:
            keisei_chiban_list = []
            if isinstance(hude_property['筆界未定構成筆'], list):
                for keisei_hitsu in hude_property['筆界未定構成筆']:
                    if '地番' in keisei_hitsu:
                        keisei_chiban_list.append(keisei_hitsu['地番'])
            elif isinstance(hude_property['筆界未定構成筆'], dict) and '地番' in hude_property['筆界未定構成筆']:
                keisei_chiban_list.append(hude_property['筆界未定構成筆']['地番'])

            if keisei_chiban_list:
                hude_property['地番'] = f"{hude_property['地番']}（{'/'.join(keisei_chiban_list)}）"
            del hude_property['筆界未定構成筆'] # 不要になった筆界未定構成筆の情報を削除

        xml_property = setExistProperties(mojObj,
                                            ['crs', 'datum_type', 'city_name'],
                                            ['座標系', '測地系判別', '市区町村名'])

        # 任意座標系のみ出力 優先
        #-----------------------------------------------
        if treated_flag is True:
            untreated_flag is False
        #-----------------------------------------------

        if untreated_flag is True:
            if ('任意座標系' in xml_property['座標系']):
                sys.exit( )

        # 任意座標系のみ出力の場合
        #-----------------------------------------------
        if treated_flag is True:
            if ('公共座標' in xml_property['座標系']):
                sys.exit( )
        #-----------------------------------------------

        sorted_properties = sortProperties(dict(**xml_property, **hude_property),
                                            ['地番',
                                             '市区町村名',
                                             '大字名',
                                             '丁目名',
                                             '小字名',
                                             '予備名',
                                             '座標系',
                                             '地図番号',
                                             '精度区分',
                                             '座標値種別',
                                             '測地系判別',
                                             '縮尺分母']) # '筆界未定構成筆' は削除したので不要

        # 地図のプロパティと各筆のプロパティをマージ
        feature = Feature(properties=sorted_properties)

        # 連続で重複している要素を削除
        exterior = [k for k, g in itertools.groupby(IdSurfaceTable[ref_gm_surface]['exterior'])]
        interior = IdSurfaceTable[ref_gm_surface]['interior']
        if len(interior) > 0:
            interior = [k for k, g in itertools.groupby(IdSurfaceTable[ref_gm_surface]['interior'])]
            for i in range(len(interior)):
                interior[i] = [k for k, g in itertools.groupby(interior[i])]

        # 測地系の変換 to 4326(WGS 84)
        if crs_name is not None:    # 任意座標は測地系の変換をしない
            if convert_flag is True: # -xオプション  座標変換しない
                crs_name = 'urn:ogc:def:crs:EPSG::' + epsg
            else:
                exterior = ConvertDatum.xy2bl(exterior, mojObj['number_crs'], 4326)
                if len(interior) > 0:
                    for i in range(len(interior)):
                        interior[i] = ConvertDatum.xy2bl(interior[i], mojObj['number_crs'], 4326)
                crs_name = 'urn:ogc:def:crs:EPSG::4326'
        else:
            crs_name = 'urn:ogc:def:crs:EPSG::' + epsg

        if len(interior) == 0:
            feature['geometry'] = {'type': 'MultiPolygon',
                                    'coordinates': [[exterior],], }
        elif len(interior) > 0:
            coord = []
            coord.append(exterior)
            for i in interior:
                coord.append(i)
            feature['geometry'] = {'type': 'MultiPolygon',
                                    'coordinates': [coord], }

        # 追加属性
        # ----------------------------------------------------------------------
        # 地図XMLファイル
        feature['properties']['地図XMLファイル'] = os.path.basename(srcFile)

        # 地番区域
        cityname = mojObj['city_name']
        oazamei = hude.get('大字名')
        if oazamei is None:
            oazamei = ''
        chomemei = hude.get('丁目名')
        if chomemei is None:
            chomemei = ''
        koazamei = hude.get('小字名')
        if koazamei is None:
            koazamei = ''
        yobimei = hude.get('予備名')
        if yobimei is None:
            yobimei = ''
        chibankuiki = cityname + oazamei + chomemei + koazamei + yobimei
        feature['properties']['地番区域'] = chibankuiki
        # ----------------------------------------------------------------------

        if daihyo_flag is True:
            feature = PointOnSurface.PointOnSurface(feature)
            lat = str(feature['properties']['代表点緯度／X座標'])
            lon = str(feature['properties']['代表点経度／Y座標'])
            daihyo_text = daihyo_text + '{"type": "Feature","properties": {"地番": "' + hude_property['地番'] + '"},"geometry": {"type": "Point","coordinates": [' + lon + ',' + lat + ']}},'  #2025～

        feature_list.append(feature)

    # 代表点を別ファイルで出力
    if daihyo_flag is True:
        daihyo_text = daihyo_text[:-1]
        daihyo_geojson_str = daihyo_text + ']}'


    # daihyoが付くgeojsonの保存
    if daihyo_flag is True:
        chiban_dir = os.path.join(os.path.dirname(src_file), 'chiban')
        os.makedirs(chiban_dir, exist_ok=True)
        dst_name_daihyo = pathlib.Path(chiban_dir) / pathlib.Path(src_file).with_suffix(".daihyo.geojson").name
        dst_name_daihyo = dst_name_daihyo.resolve()
        if daihyo_geojson_str: # daihyoデータが存在する場合のみ保存
            daihyo_geojson_bytes = daihyo_geojson_str.encode('utf-8')
            if len(daihyo_geojson_bytes) >= 140:
                with open(dst_name_daihyo, 'w', encoding='utf-8') as f:
                    f.write(daihyo_geojson_str)

    feature_collection = FeatureCollection(feature_list)
    if crs_name is not None:
        feature_collection['crs'] = {'properties': {'name': crs_name}, 'type': 'name'}
    else:
        feature_collection['crs'] = crs_name

    return geojson.dumps(feature_collection, ensure_ascii=False), daihyo_geojson_str


def setExistProperties(obj, from_key_list, to_key_list):
    tmp_obj = {}
    for key in from_key_list:
        if key in obj.keys():
            i = from_key_list.index(key)
            tmp_obj[to_key_list[i]] = obj[key]
        else:
            i = from_key_list.index(key)
            tmp_obj[to_key_list[i]] = None

    return tmp_obj


def sortProperties(obj, sort_key_list):
    tmp_obj = {}
    for key in sort_key_list:
        if key in obj.keys():
            i = sort_key_list.index(key)
            tmp_obj[sort_key_list[i]] = obj[key]
        else:
            i = sort_key_list.index(key)
            tmp_obj[sort_key_list[i]] = None

    return tmp_obj
